/***************************************************************************
                          main.c  -  read algorithms
                             -------------------

    bkread - utility to read BK tapes via soundcard

    begin                : Sun Oct 28 2001 02:51:53 MSD
    copyright            : (C) 2001 by Alexander "las" Lunev
    email                : imbecyle@mail.ru
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bkread.h"
#include "audio.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stropts.h>
#include <string.h>
#include <signal.h>

int verbose = 0;

char *last_error = NULL;
int last_pattern = 0;

unsigned char data[65536];

void usage(char *arg) {
    fprintf(stderr, "Argument: %s\n", arg);
    fprintf(stderr,
"Usage: bkread [-v] [-r|m|e|s|R|a num] [-p 0|1] [-A file]%s [name]\n\
Where name is the name of file to read, the switches are:\n\
 -v - verbose mode on\n\
 -r num - set recording volume (0..100)\n\
 -m num - set monitor volume (0..100)\n\
 -e num - set edge level (0..255), default is 128\n\
 -s num - set starting SYNC pattern count (detect SYNC)\n\
 -p 0|1 - set pattern start edge to falling (0) or raising (1)\n\
 -R num - set the sampling rate in Knum/sec (%d..%d) or num/sec (%d..%d)\n\
          default is %d\n\
 -a num - set maximum count of speed adaptations, default is unlimited\n\
 -A file - use specified audio device (default \"%s\")\n%s%s%s",
#if defined (USE_LINUX_AUDIO)
            " [-M file]",
#else
            "",
#endif
            MIN_SAMPLE_RATE_K, MAX_SAMPLE_RATE_K,
            MIN_SAMPLE_RATE, MAX_SAMPLE_RATE, DEFAULT_SAMPLE_RATE,
            audio_name,
#if defined (USE_LINUX_AUDIO)
            " -M file - use specified mixer device (default \"", mixer_name, "\")\n"
#else
            "", "", ""
#endif
    );
    exit(1);
}

void cancel(int sig) {
    closeall();
    signal(sig, SIG_DFL);
    if(verbose) fprintf(stderr, "Signal (%d) received. Program stopped.\n", sig);
    exit(0);
}

int main(int ac, char *av[]) {
    int i, nb;
    int val;
    int data_checksum, data_length, data_addr;
    char file_name[32];
    struct stat dummy_stat;
    unsigned int checksum;
    char *search_file = NULL;
    int dummy_read = 0;
    int new_monitor_gain = -1, new_record_gain = -1, new_sample_rate = -1;
    
    for(i = 1; i < ac; i++) {
        if(*av[i] == '-') {
          if(strlen(av[i]) != 2) usage(av[i]);
          switch(*(av[i] + 1)) {
            case 'r':
                if(++i >= ac) usage("-r without value");
                val = atoi(av[i]);
                if((val < 0) || (val > 100))
                  usage("-r num - out of range, must be in range 0..100");
                new_record_gain = val;
                break;
            case 'm':
                if(++i >= ac) usage("-m without value");
                val = atoi(av[i]);
                if((val < 0) || (val > 100))
                  usage("-m: num out of range, must be in range 0..100");
                new_monitor_gain = val;
                break;
            case 'R':
                if(++i >= ac) usage("-R without value");
                val = atoi(av[i]);
                if((val < MIN_SAMPLE_RATE_K) ||
                   ((val > MAX_SAMPLE_RATE_K) && (val < MIN_SAMPLE_RATE)) ||
                   (val > MAX_SAMPLE_RATE)
                  )
                  usage("-R: num is out of range");
                new_sample_rate = val >= MIN_SAMPLE_RATE ? val : val * 1000;
                break;
            case 'e':
                if(++i >= ac) usage("-e without value");
                val = atoi(av[i]);
                if((val < 0) || (val > 255))
                  usage("-e: num out of range, must be in range 0..255");
                edge_level = val;
                break;
            case 's':
                if(++i >= ac) usage("-s without value");
                val = atoi(av[i]);
                if(val <= 10)
                  usage("-s: num less than ten");
                pattern_count = val;
                break;
            case 'p':
                if(++i >= ac) usage("-p without value");
                val = atoi(av[i]);
                if((val < 0) || (val > 1))
                  usage("-s: num must be 0 or 1");
                pattern_start = val;
                break;
            case 'a':
                if(++i >= ac) usage("-a without value");
                val = atoi(av[i]);
                max_adaptations = val;
                break;
            case 'v':
                verbose = 1;
                break;
            case 'A':
                if(++i >= ac) usage("-A without device file");
                audio_name = av[i];
                break;
#if defined (USE_LINUX_AUDIO)
            case 'M':
                if(++i >= ac) usage("-M without device file");
                mixer_name = av[i];
                break;
#endif
            default:
                usage(av[i]);
          }
	} else {
	    if(search_file != NULL) usage(av[i]);
	    search_file = av[i];
	}
    }

    open_audio();

    signal(SIGHUP, &cancel);
    signal(SIGINT, &cancel);
    signal(SIGTERM, &cancel);

    init_audio();

    if(new_monitor_gain >= 0) set_monitor_gain(new_monitor_gain);
    if(new_record_gain >= 0) set_rec_gain(new_record_gain);
    if(new_sample_rate >= 0) set_sample_rate(new_sample_rate);

    if(verbose) {
        fprintf(stderr, "Audio:    recording gain = %d\n", record_gain);
        fprintf(stderr, "Audio:    monitor gain = %d\n", monitor_gain);
        fprintf(stderr, "Audio:    sample_rate = %d\n", sample_rate);
        fprintf(stderr, "Audio:    edge level = %d\n", edge_level);
        fprintf(stderr, "Analyzer: synchro tuning pattern count = %d\n", pattern_count);
        if(max_adaptations) fprintf(stderr, "Analyzer: Max. number of adaptations = %d\n", max_adaptations);
	else fputs("Analyzer: max. number of adaptations set to unlimited\n", stderr);
    }
//
// Waiting for a file header synchros
//
    for(;;) {
        detect_preamble();
        if(detect_marker() < 0) {
            if(verbose)
                fprintf(stderr, "Can't detect marker: %s.\n", last_error);
            continue;
        }

        for(i = 0; i < 20; i++)
            if(read_byte(&data[i]) < 0) {
                if(verbose) fprintf(stderr, "Error in header (%s); offset = %o (octal).\n", last_error, i);
                break;
            }
        if(i != 20) continue;

        data_addr = data[0] + (data[1] << 8);
        data_length = data[2] + (data[3] << 8);
        memcpy(file_name, &data[4], 16);
        file_name[16] = '\0';
        for(i = 15; i >= 0; i--)
            if(file_name[i] != ' ') break;
            else file_name[i] = '\0';

        if(verbose) {
            fprintf(stderr, "File Name = \"%-16s\"\n", file_name);
            fprintf(stderr, "Data Length = %6o (octal).\n", data_length);
            fprintf(stderr, "Data Address = %6o (octal).\n", data_addr);
        } else fprintf(stderr, "%s\n", file_name);
	
        if(search_file != NULL) {
					if(strcmp(search_file, file_name)) dummy_read = 1; else dummy_read = 0;
				}

        if((val = read_bit()) != 0 ||
           (val = read_bit()) != 0 ||
           (val = read_bit()) != 0 ||
           (val = read_bit()) != 0) {
            if(verbose)
                fprintf(stderr, "Can't read finishing \"0000\" at the end of header (%d).\n", val);
            continue;
        }

        if((i = analyze_pattern(read_pattern(max_period))) != PATTERN_SYNC) {
            if(verbose) {
                fprintf(stderr, "Can't find SYNC at the end of header (%d)", i);
                if(i < 0) fprintf(stderr, " - %s.\n", last_error);
                else fprintf(stderr, ".\n");
            }
            continue;
        }

        if((i = read_bit()) != 1) {
            if(verbose) {
                fprintf(stderr, "Header doesn't finish by \"1\" bit (%d)", i);
                if(i < 0) fprintf(stderr, " - %s.\n", last_error);
                else fprintf(stderr, ".\n");
            }
            continue;
        }

        for(nb = 20; nb < data_length + 22; nb++) {
            if(read_byte(&data[nb]) < 0) {
                if(nb >= (data_length + 20)) {
                    fprintf(stderr, "Error reading checksum byte %6o (octal): %s.\n", nb - 20 - data_length, last_error);
                    nb = data_length + 22;
                    data[nb - 2] = data[nb - 1] = 0;
                } else {
                    fprintf(stderr, "Error reading data byte %6o (octal): %s.\n", nb - 20, last_error);
                }

                break;
            }
        }
        if(nb != data_length + 22) continue;

        data_checksum = data[nb - 2] + (data[nb - 1] << 8);
        if(verbose) {
            fprintf(stderr, "Tape Checksum = %6o (octal).\n", data_checksum);
        }

        for(checksum = 0, i = 20; i < nb - 2; i++) {
            checksum += data[i];
            checksum += (checksum & 0200000) != 0;
            checksum &= 0177777;
        }

        if(verbose) {
            fprintf(stderr, "Data Checksum = %6o (octal).\n", checksum);
        }
        if(checksum != data_checksum) {
            fprintf(stderr, "Warning: checksum error.\n");
        }

        if(!dummy_read) {
    	    val = strlen(file_name);
    	    for(i = 1; stat(file_name, &dummy_stat) >= 0; i++) {
        	sprintf(&file_name[val], ".%d", i);
    	    }

    	    if(verbose) fprintf(stderr, "Saving file to: %s (%d bytes).\n", file_name, nb);

    	    if(((i = open(file_name, O_WRONLY | O_CREAT, 0644)) < 0) ||
               (write(i, data, nb) != nb)) {
        	fprintf(stderr, "Can't save file %s; ", file_name);
        	perror("open()|write()");
    	    } else if(!verbose) fprintf(stderr, "%d bytes written.\n", nb);
    	    close(i);
			}
   }
   exit(0);
}
