/*
**
** (C)1988 Software for RT-11.5 by LAS & Co Inc.
**  -- K&R style code DECUS C
** (C)2000 Unix port (for bkread) by Alexander "las" Lunev
**  -- ANSI C (gcc/cc)
** (C)2000 Analysis modules (for bkread) by Alexander "las" Lunev
**
** Simple Disassembler for KR1808VM1 CPU instructions.
** v01.5.1 (BK version)
*/
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <string.h>
#include <stdlib.h>

static char *usage="Usage: dasm [-a | -b | -c | -f forced_file] tape_file\n";

void error(const char *s) {
    fputs(s, stderr);
    exit(1);
}

static int infile;
static unsigned short word[4];
static unsigned short last_addr;
static int nwords;
static int read_status = 1;

/* Processing flags :                                                       */

static int ascflg=0;        /* Indicate that ASCII codes needed (-a flag)   */
static int bytflg=0;        /* Indicate that byte codes needed (-b flag)    */
static int wrdflg=1;        /* Indicate that program codes needed (-c flag) */

static int unsigned addr=0;
static int unsigned start=0;

static char outstring[512];
#define append() (&outstring[strlen(outstring)])
#define do_out() (fputs(outstring, stdout), outstring[0] = '\0')

#define OTHER 1
#define BINARY 2
#define UNARY 3
#define RDD 4
#define XXX 5
#define MARK 6
#define ILLEGAL 0

static struct cmd_tbl {
  unsigned short mask;
  int type;
} codes[] = {
        { 06, OTHER },
        { 077, ILLEGAL },
        { 0207, UNARY },
        { 0277, OTHER },
        { 0377, UNARY },
        { 03777, XXX },
        { 04777, RDD },
        { 06377, UNARY },
        { 06477, MARK },
        { 06677, ILLEGAL },
        { 06777, UNARY },
        { 07777, ILLEGAL },
        { 067777, BINARY },
        { 077777, RDD },
        { 0104777, XXX },
        { 0106577, UNARY },
        { 0107777, ILLEGAL },
        { 0167777, BINARY },
        { 0, 0 }
};

unsigned short getword(int do_append) {
    unsigned char c;
    unsigned short wd = 0;

    if(!do_append) nwords = 0;
    addr += 2;
    if((read_status = read(infile, &c, 1)) <= 0) return word[nwords++] = 0;
    wd = c;
    if((read_status = read(infile, &c, 1)) <= 0) return word[nwords++] = wd;
    return word[nwords++] = (short)(wd | (c << 8));
}

int type_cmd(unsigned short rcod) {
 int i;
 
 for(i = 0; codes[i].mask != 0; i++)
  if(rcod <= codes[i].mask) return(codes[i].type);
 return ILLEGAL;
}

#define MAX_REF 32868
#define MAX_FORCED 8192
static unsigned short references[MAX_REF];    
static unsigned short unreacheables[MAX_REF];
static unsigned short forced_code[MAX_FORCED];
static unsigned short forced_data[MAX_FORCED];
static unsigned short symb_addr[MAX_FORCED];
static char* forced_code_labels[MAX_FORCED];
static char* forced_data_labels[MAX_FORCED];
static char* symb_labels[MAX_FORCED];
static int ref_count = 0, unref_count = 0;
static int code_count = 0, data_count = 0;
static int symb_count = 0;

void add_ref(unsigned short addr) {
    int i, j;
    
    if(!ref_count) {
        references[0] = addr;
        ref_count = 1;
    } else {
        if(ref_count >= MAX_REF) error("Too many references.\n");
        for(i = 0; i < ref_count; i++)
            if(references[i] > addr) {
                for(j = ref_count; j > i; j--)
                    references[j] = references[j - 1];
                break;
            } else if(references[i] == addr) return;
        references[i] = addr;
        ref_count++;
    }
}

void add_unreach(unsigned short addr) {
    int i, j;
    
    if(!unref_count) {
        unreacheables[0] = addr;
        unref_count = 1;
    } else {
        if(ref_count >= MAX_REF) error("Too many unreacheable fragments.\n");
        for(i = 0; i < unref_count; i++)
            if(unreacheables[i] > addr) {
                for(j = unref_count; j > i; j--)
                    unreacheables[j] = unreacheables[j - 1];
                break;
            } else if(unreacheables[i] == addr) return;
        unreacheables[i] = addr;
        unref_count++;
    }
}

void add_code(const unsigned short addr, const char* label) {
    int i, j;
    
    if(!code_count) {
        forced_code[0] = addr;
        if((forced_code_labels[0] = malloc(8)) == NULL) error("Can't allocate memory (AC0)\n");
        strncpy(forced_code_labels[0], label, 8);
        *(forced_code_labels[0] + 7) = '\0';
        code_count = 1;
    } else {
        if(code_count >= MAX_FORCED) error("Too many forced code fragments.\n");
        for(i = 0; i < code_count; i++)
            if(forced_code[i] > addr) {
                for(j = code_count; j > i; j--) {
                    forced_code[j] = forced_code[j - 1];
                    forced_code_labels[j] = forced_code_labels[j - 1];
                }
                break;
            }
        forced_code[i] = addr;
        if((forced_code_labels[i] = malloc(8)) == NULL) error("Can't allocate memory (AC)\n");
        strncpy(forced_code_labels[i], label, 8);
        *(forced_code_labels[i] + 7) = '\0';
        code_count++;
    }
}

void add_data(const unsigned short addr, const char* label) {
    int i, j;
    
    if(!data_count) {
        forced_data[0] = addr;
        if((forced_data_labels[0] = malloc(8)) == NULL) error("Can't allocate memory (AD0)\n");
        strncpy(forced_data_labels[0], label, 8);
        *(forced_data_labels[0] + 7) = '\0';
        data_count = 1;
    } else {
        if(data_count >= MAX_REF) error("Too many forced data fragments.\n");
        for(i = 0; i < data_count; i++)
            if(forced_data[i] > addr) {
                for(j = data_count; j > i; j--) {
                    forced_data[j] = forced_data[j - 1];
                    forced_data_labels[j] = forced_data_labels[j - 1];
                }
                break;
            }
        forced_data[i] = addr;
        if((forced_data_labels[i] = malloc(8)) == NULL) error("Can't allocate memory (AD)\n");
        strncpy(forced_data_labels[i], label, 8);
        *(forced_data_labels[i] + 7) = '\0';
        data_count++;
    }
}

void add_symb(const unsigned short addr, const char* label) {
    int i, j;
    
    if(!symb_count) {
        symb_addr[0] = addr;
        if((symb_labels[0] = malloc(8)) == NULL) error("Can't allocate memory (AS0)\n");
        strncpy(symb_labels[0], label, 8);
        *(symb_labels[0] + 7) = '\0';
        symb_count = 1;
    } else {
        if(symb_count >= MAX_REF) error("Too many symbols.\n");
        for(i = 0; i < symb_count; i++)
            if(symb_addr[i] > addr) {
                for(j = symb_count; j > i; j--) {
                    symb_addr[j] = symb_addr[j - 1];
                    symb_labels[j] = symb_labels[j - 1];
                }
                break;
            }
        symb_addr[i] = addr;
        if((symb_labels[i] = malloc(8)) == NULL) error("Can't allocate memory (AS)\n");
        strncpy(symb_labels[i], label, 8);
        *(symb_labels[i] + 7) = '\0';
        symb_count++;
    }
}

unsigned short get_op_addr(int code) {
 int reg, type;
 
 reg = code & 07;
 type = (code & 070) >> 3;
 if(reg != 7 || type == 0 || type == 1 || type == 4 || type == 5)
  switch(type) {
    case 6:
    case 7:    getword(1);
               return 0;
    default:   return 0;
  }
 else
  switch(type) {
    case 3:    return getword(1);
    case 6:    return (unsigned short)(getword(1) + addr);
    default:   getword(1); return 0;
  }
}

void analyze1_regdd(unsigned short rcod) {
 unsigned short cmd;
 
 cmd = rcod & 077000;
 if(cmd == 04000) add_ref(get_op_addr(rcod & 077));          // JSR
 else if(cmd == 077000) add_ref(addr - ((rcod & 077) << 1)); // SOB
}

void analyze1_binary(unsigned short code) { /* Binary instructions */
 switch((code & 070000) >> 12)              /* Select opcode */
  {
   case 1:
   case 2:
   case 3:
   case 4:
   case 5:
   case 6:
        get_op_addr((code & 07700) >> 6);
        get_op_addr(code & 077);
  }
}

void analyze1_unary(unsigned rcod) {
 switch((rcod & 07700) >> 6)
  {
   case 001:    // JMP
        add_ref(get_op_addr(rcod & 077));
        add_unreach(addr);
        break;
   case 002:    // RTS
        add_unreach(addr);
        break;
   case 003:
   case 050:
   case 051:
   case 052:
   case 053:
   case 054:
   case 055:
   case 056:
   case 057:
   case 060:
   case 061:
   case 062:
   case 063:
   case 064:
   case 065:
   case 067:
        get_op_addr(rcod & 077);
        break;
  }
}

void analyze1_codxxx(unsigned rcod) {
 unsigned short raddr;

 if((rcod & 0177000) == 0104000) {                  /* EMT & TRAP codes */
   return;
 }

 switch((rcod & 0177400) >> 6) {
   case 04:     // BR
        add_unreach(addr);
   case 010:
   case 014:
   case 020:
   case 024:
   case 030:
   case 034:
   case 01000:
   case 01004:
   case 01010:
   case 01014:
   case 01020:
   case 01024:
   case 01030:
   case 01034:
        raddr = (rcod & 0200) ? rcod | 0177400 : rcod & 0377;
        raddr <<= 1;
        add_ref(raddr + addr);
 }
}

void analyze1_othercmd(unsigned rcod) {
 switch(rcod)
  {
   case 00: // HALT
   case 02: // RTI
   case 06: // RTT
        add_unreach(addr);
  }
}

void comment() {
 int i, l;
 
 for(i = l = 0; outstring[i]; i++) if(outstring[i] == '\n') l = 0; else l++;
 for(; l < 40; l++, i++) outstring[i] = ' ';
 outstring[i] = '\0';
 
 sprintf(append(), "!%06o: ", last_addr);

 if(wrdflg)
  for(i = 0; i < nwords; i++) sprintf(append(), "%06o ", word[i]);
 
 if(bytflg)
  for(i = 0; i < nwords; i++) {
    sprintf(append(), "%03o %03o ", word[i] & 0377, (word[i] >> 8) & 0377);
  }

 if(ascflg) {
  strcat(outstring, "\"");
  for(i = 0; i < nwords; i++) {
    sprintf(append(), "%c%c",
            (word[i] & 0377) >= ' ' ? (word[i] & 0377) : '.',
            ((word[i] >> 8) & 0377) >= ' ' ? (word[i] >> 8) & 0377 : '.');
  }
  strcat(outstring, "\"");
 }

 strcat(outstring, "\n");
}

static char *code_label_name(unsigned short addr) {
    static char buff[16];
    int i;
    
    for(i = 0; (i < code_count) && (forced_code[i] < addr); i++);
    
    if((i < code_count) && forced_code[i] == addr) strcpy(buff, forced_code_labels[i]);
    else {
        for(i = 0; (i < symb_count) && (symb_addr[i] < addr); i++);
    
        if((i < symb_count) && symb_addr[i] == addr) strcpy(buff, symb_labels[i]);
        else sprintf(buff, "%o", addr);
    }
    
    return buff;
}

static char *data_label_name(unsigned short addr) {
    static char buff[16];
    int i;
    
    for(i = 0; (i < data_count) && (forced_data[i] < addr); i++);
    
    if((i < data_count) && forced_data[i] == addr) strcpy(buff, forced_data_labels[i]);
    else {
        for(i = 0; (i < symb_count) && (symb_addr[i] < addr); i++);
    
        if((i < symb_count) && symb_addr[i] == addr) strcpy(buff, symb_labels[i]);
        else strcpy(buff, code_label_name(addr));
    }
    
    return buff;
}

static char *register_name[] = {
    "R0", "R1", "R2", "R3", "R4", "R5", "SP", "PC"
};

char *regname(int reg) {
    return register_name[reg];
}

void data_outmet(int code) {
 int reg, type;
 
 reg = code & 07;
 type = (code & 070) >> 3;
 if(reg != 7 || type == 0 || type == 1 || type == 4 || type == 5)
  switch(type) {
    case 0:    sprintf(append(), "%s", regname(reg));
               break;
    case 1:    sprintf(append(), "(%s)", regname(reg));
               break;
    case 2:    sprintf(append(), "(%s)+", regname(reg));
               break;
    case 3:    sprintf(append(), "@(%s)+", regname(reg));
               break;
    case 4:    sprintf(append(), "-(%s)", regname(reg));
               break;
    case 5:    sprintf(append(), "@-(%s)", regname(reg));
               break;
    case 6:    sprintf(append(), "%s(%s)", data_label_name(getword(1)), regname(reg));
               break;
    case 7:    sprintf(append(), "@%s(%s)", data_label_name(getword(1)), regname(reg));
               break;
  }
 else
  switch(type) {
    case 2:    sprintf(append(), "#%s", data_label_name(getword(1)));
               break;
    case 3:    sprintf(append(), "@#%s", data_label_name(getword(1)));
               break;
    case 6:    sprintf(append(), "%s", data_label_name((unsigned short)(getword(1) + addr)));
               break;
    case 7:    sprintf(append(), "@%s", data_label_name((unsigned short)(getword(1) + addr)));
               break;
  }
}

void code_outmet(int code) {
 int reg, type;
 
 reg = code & 07;
 type = (code & 070) >> 3;
 if(reg != 7 || type == 0 || type == 1 || type == 4 || type == 5)
  switch(type) {
    case 0:    sprintf(append(), "%s", regname(reg));
               break;
    case 1:    sprintf(append(), "(%s)", regname(reg));
               break;
    case 2:    sprintf(append(), "(%s)+", regname(reg));
               break;
    case 3:    sprintf(append(), "@(%s)+", regname(reg));
               break;
    case 4:    sprintf(append(), "-(%s)", regname(reg));
               break;
    case 5:    sprintf(append(), "@-(%s)", regname(reg));
               break;
    case 6:    sprintf(append(), "%s(%s)", code_label_name(getword(1)), regname(reg));
               break;
    case 7:    sprintf(append(), "@%s(%s)", code_label_name(getword(1)), regname(reg));
               break;
  }
 else
  switch(type) {
    case 2:    sprintf(append(), "#%s", code_label_name(getword(1)));
               break;
    case 3:    sprintf(append(), "@#%s", code_label_name(getword(1)));
               break;
    case 6:    sprintf(append(), "%s", code_label_name((unsigned short)(getword(1) + addr)));
               break;
    case 7:    sprintf(append(), "@%s", code_label_name((unsigned short)(getword(1) + addr)));
               break;
  }
}

void regdd(unsigned short rcod) {
 unsigned short cmd, reg;
 
 reg = (rcod & 0700) >> 6;
 cmd = rcod & 077000;
 if(cmd == 04000) { sprintf(append(), "JSR     %s, ", regname(reg)); code_outmet(rcod & 077); }
 else if(cmd == 077000) sprintf(append(), "SOB     %s, %s", regname(reg), code_label_name((unsigned short)(addr - ((rcod & 077) << 1))));
 else if(cmd==074000) { sprintf(append(), "XOR     %s, ", regname(reg)); data_outmet(rcod & 077); }
 else sprintf(append(), ".WORD   %o !E-R", rcod);
}

void outcmd(char *name, unsigned short code) {
 int i = strlen(name);
 
 strcat(outstring, name);
 if(code & 0100000) { strcat(outstring, "B"); i++; }
 for(;i < 8; i++) strcat(outstring, " ");
}

void binary(unsigned short code) {          /* Binary instructions */
 switch((code & 070000) >> 12)              /* Select opcode */
  {
   case 1:
        outcmd("MOV", code);
        break;
   case 2:
        outcmd("CMP", code);
        break;
   case 3:
        outcmd("BIT", code);
        break;
   case 4:
        outcmd("BIC", code);
        break;
   case 5:
        outcmd("BIS", code);
        break;
   case 6:
        if(code & 0100000)
         sprintf(append(), "ADD     ");
        else sprintf(append(), "SUB     ");
        break;
   default:
        sprintf(append(), ".WORD   %o !E-B", code);
        return;
  }
 data_outmet((code & 07700) >> 6);
 strcat(outstring, ", ");
 data_outmet(code & 077);
}

void unary(unsigned rcod) {
 switch((rcod & 07700) >> 6)
  {
   case 001:    strcat(outstring, "JMP     "); break;
   case 002:    strcat(outstring, "RTS     "); break;
   case 003:    strcat(outstring, "SWAB    "); break;
   case 050:    outcmd("CLR", rcod); break;
   case 051:    outcmd("COM", rcod); break;
   case 052:    outcmd("INC", rcod); break;
   case 053:    outcmd("DEC", rcod); break;
   case 054:    outcmd("NEG", rcod); break;
   case 055:    outcmd("ADC", rcod); break;
   case 056:    outcmd("SBC", rcod); break;
   case 057:    outcmd("TST", rcod); break;
   case 060:    outcmd("ROR", rcod); break;
   case 061:    outcmd("ROL", rcod); break;
   case 062:    outcmd("ASR", rcod); break;
   case 063:    outcmd("ASL", rcod); break;
   case 064:    strcat(outstring, "MTPS    "); break;
   case 065:    strcat(outstring, "MFPS    "); break;
   case 067:    strcat(outstring, "SXT     "); break;
   default: sprintf(append(), ".WORD   %o !E-U", rcod); return; /* Here may be illegal command */
  }
 if((rcod & 07700) >> 6 == 1) code_outmet(rcod & 077);
 else data_outmet(rcod & 077);
}

void codxxx(unsigned rcod) {
 unsigned short raddr;

 if((rcod & 0177000) == 0104000) {                  /* EMT & TRAP codes */
   strcat(outstring, (rcod & 0177400) == 0104000 ? "EMT " : "TRAP");
   sprintf(append(), "    %o", rcod & 0377);
   return;
 }

 raddr = (rcod & 0200) ? rcod | 0177400 : rcod & 0377;
 raddr <<= 1;
 raddr += addr;
 switch((rcod & 0177400) >> 6) {
   case 04:  sprintf(append(), "BR      %s", code_label_name(raddr)); break;
   case 010: sprintf(append(), "BNE     %s", code_label_name(raddr)); break;
   case 014: sprintf(append(), "BEQ     %s", code_label_name(raddr)); break;
   case 020: sprintf(append(), "BGE     %s", code_label_name(raddr)); break;
   case 024: sprintf(append(), "BLT     %s", code_label_name(raddr)); break;
   case 030: sprintf(append(), "BGT     %s", code_label_name(raddr)); break;
   case 034: sprintf(append(), "BLE     %s", code_label_name(raddr)); break;
   case 01000: sprintf(append(), "BPL     %s", code_label_name(raddr)); break;
   case 01004: sprintf(append(), "BMI     %s", code_label_name(raddr)); break;
   case 01010: sprintf(append(), "BHI     %s", code_label_name(raddr)); break;
   case 01014: sprintf(append(), "BLOS    %s", code_label_name(raddr)); break;
   case 01020: sprintf(append(), "BVC     %s", code_label_name(raddr)); break;
   case 01024: sprintf(append(), "BVS     %s", code_label_name(raddr)); break;
   case 01030: sprintf(append(), "BHIS    %s", code_label_name(raddr)); break;
   case 01034: sprintf(append(), "BLO     %s", code_label_name(raddr)); break;
   default:    sprintf(append(), ".WORD   %o !E-X", rcod);   /* Here may be illegal command */
 }
}

void othercmd(unsigned rcod) {
 switch(rcod)
  {
   case 0240: strcat(outstring, "NOP     "); break;
   case 0241: strcat(outstring, "CLC     "); break;
   case 0242: strcat(outstring, "CLV     "); break;
   case 0244: strcat(outstring, "CLZ     "); break;
   case 0243: strcat(outstring, "CCC     "); break;
   case 0250: strcat(outstring, "CLN     "); break;
   case 0261: strcat(outstring, "SEC     "); break;
   case 0262: strcat(outstring, "SEV     "); break;
   case 0264: strcat(outstring, "SEZ     "); break;
   case 0270: strcat(outstring, "SEN     "); break;
   case 0277: strcat(outstring, "SCC     "); break;
   case 00:   strcat(outstring, "HALT    "); break;
   case 01:   strcat(outstring, "WAIT    "); break;
   case 02:   strcat(outstring, "RTI     "); break;
   case 03:   strcat(outstring, "BPT     "); break;
   case 04:   strcat(outstring, "IOT     "); break;
   case 05:   strcat(outstring, "RESET   "); break;
   case 06:   strcat(outstring, "RTT     "); break;
   default:   sprintf(append(), ".WORD   %o !E-O", rcod);   /* Here may be illegal command */
  }
}

void outword(unsigned code) {
 sprintf(append(), ".WORD   %o", code);
}

void read_forced(char *file) {
    FILE *f;
    char c;
    char l[8];
    int a;
    
    if((f = fopen(file, "r")) == NULL) error("Can't open forced file.\n");
    while(!feof(f)) {
        if(fscanf(f, "%c %o %7s\n", &c, &a, l) != 3) error("Error in forced file.\n");
        switch(c) {
            case 'C':
            case 'c':
                add_code((unsigned short)a, l);
                break;
            case 'D':
            case 'd':
                add_data((unsigned short)a, l);
                add_unreach((unsigned short)a);
                break;
            case 'S':
            case 's':
                add_symb((unsigned short)a, l);
                break;
        }
    }
    fclose(f);
}

int main(int argc, char *argv[]) {
 unsigned short code;
 int fc_index, fd_index, ref_index, unreach_index;
 int i, need_cr;
 char buff[64];

 for(i = 1; i < argc; i++)
  if(*argv[i] != '-' || *(argv[i] + 2) != '\0') break;
   else switch(*(argv[i] + 1)) {
      case 'a': ascflg=1; break;
      case 'b': bytflg=1; break;
      case 'c': wrdflg=0; break;
      case 'f': if(++i < argc) read_forced(argv[i]);
                else error(usage);
                break;
      default : error(usage);
    }
  
  if(i != (argc - 1)) error(usage);
  if((infile = open(argv[i], O_RDONLY)) < 0) {
    perror(argv[i]);
    error("dasm: can't open input file. Exiting.\n");
  }

  outstring[0] = '\0';
  do_out();
  sprintf(append(), "; Disassembler V01.5.1 BK 001x-xx (bkread)\n; Alexander \"las\" Lunev\n");
  add_code(start = getword(0), "$START$");
  sprintf(append(), "; Start address = %o\n; Length = %o\n", start, getword(0));
  
  if(read(infile, buff, 16) != 16) error("Error reading header\n");
  buff[16] = '\0';
  sprintf(append(), "; Tape file name: %-16s\n", buff);
  sprintf(append(), ";       Command Operand(s)              Address  Codes\n;\n");
  do_out();
  
  for(addr = start; read_status > 0;) {
   last_addr = addr;
   code = getword(0);
   switch(type_cmd(code)) {              /*  Select command type  */
    case RDD:
        analyze1_regdd(code);
        break;
    case XXX:
        analyze1_codxxx(code);
        break;
    case OTHER:
        analyze1_othercmd(code);
        break;
    case BINARY:
        analyze1_binary(code);
        break;
    case UNARY:
        analyze1_unary(code);
        break;
    case MARK:
        break;
    default :
   }
   outstring[0] = nwords = 0;
  }

  if(read_status < 0) {
     perror("Step 1, read()");
     error("Can't read input file\n");
  }

  if(lseek(infile, 20L, SEEK_SET) != 20L) {
     perror("Step 2, lseek()");
     error("Can't seek file\n");
  }
  read_status = 1;

  add_unreach((unsigned short)0177777);
  add_code((unsigned short)0177777, "177777");
  add_data((unsigned short)0177777, "177777");
  fc_index = fd_index = ref_index = unreach_index = 0;
  
  for(addr = start; read_status > 0;) {
   last_addr = addr;
   need_cr = 0;

   for(;addr == references[ref_index]; ref_index++) {
    sprintf(append(), "%06o: ", addr);
    need_cr = 1;
    while(addr >= unreacheables[unreach_index]) unreach_index++;
   }
   for(;addr == forced_code[fc_index]; fc_index++) {
    if(need_cr) strcat(outstring, "\n");
    sprintf(append(), "%7.7s:", forced_code_labels[fc_index]);
    need_cr = 1;
    while(addr >= unreacheables[unreach_index]) unreach_index++;
   }
   for(;addr == forced_data[fd_index]; fd_index++) {
    if(need_cr) strcat(outstring, "\n");
    sprintf(append(), "%7.7s:", forced_data_labels[fd_index]);
    need_cr = 1;
   }
   
   if(!need_cr) strcat(outstring, "        ");

   while(references[ref_index] <= addr) ref_index++;
   while(forced_code[fc_index] <= addr) fc_index++;
   while(forced_data[fd_index] <= addr) fd_index++;
   
   if(addr >= unreacheables[unreach_index]) {
    outword(code = getword(0));
   } else {
    switch(type_cmd(code = getword(0))) {
        case RDD:
            regdd(code);
            break;
        case XXX:
            codxxx(code);
            break;
        case OTHER:
            othercmd(code);
            break;
        case BINARY:
            binary(code);
            break;
        case UNARY:
            unary(code);
            break;
        case MARK:
            sprintf(append(), "MARK    %o", code & 077);  /*  Command MARK */
            break;
        default :
            outword(code);
    }
   }
   comment();
   do_out();
  }

  if(read_status < 0)
     perror("Warning - file read error");

  sprintf(append(), "        .END    %o\n", start);
  do_out();
  exit(0);
}
