/***************************************************************************
                          audio.c  -  platform-depended sound handling
                             -------------------

    bkread - utility to read BK tapes via soundcard

    begin                : Sun Oct 28 2001
    copyright            : (C) 2001 by Alexander "las" Lunev
    email                : imbecyle@mail.ru
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "bkread.h"
#include "audio.h"

#if defined (SYSTEM_UNIX)
	#include <stdio.h>
	#include <unistd.h>
	#include <stdlib.h>
	#include <sys/types.h>
	#include <sys/stat.h>
	#include <fcntl.h>
        #include <errno.h>
#if defined (SYSTEM_LINUX)
	#include <sys/ioctl.h>
#else
        #include <ioctl.h>
#endif
#endif

#if defined (USE_SOLARIS_AUDIO)
// For Solaris use Sun default sound drivers
#include <sys/audioio.h>
#define AUDIO_DEVICE "/dev/audio"
#endif

#if defined (USE_LINUX_AUDIO)
// For Linux assume OSS compatible
#include <sys/soundcard.h>
#define AUDIO_DEVICE "/dev/dsp"
#define MIXER_DEVICE "/dev/mixer"
#endif

#if defined (USE_LINUX_AUDIO)
char *mixer_name = MIXER_DEVICE;
int mixerdev = -1;
int mixer_monitor_n = -1;
int mixer_reclev_n = -1;
char *devlabels[] = SOUND_DEVICE_LABELS;
#endif
char *audio_name = AUDIO_DEVICE;
int audiodev = -1;
int sample_rate = DEFAULT_SAMPLE_RATE;
int edge_level = 0x80;
int record_gain, monitor_gain = 0;

void open_audio() {
    struct stat statbuf;

    if(verbose)
        fprintf(stderr, "Opening %s as sound device...\n", audio_name);

    if(stat(audio_name, &statbuf) < 0) {
        fprintf(stderr, "Can't access %s\n", audio_name);
        exit(1);
    }
    if(!(statbuf.st_mode & S_IFCHR)) {
        fprintf(stderr, "%s Is not a character device\n", audio_name);
        exit(1);
    }
    if(((audiodev = open(audio_name, O_RDONLY | O_NDELAY)) < 0) ||
       (fcntl(audiodev, F_SETFL, O_RDONLY) < 0)) {
        perror(audio_name);
        exit(1);
    }
#if defined (USE_LINUX_AUDIO)
    if(verbose)
        fprintf(stderr, "Opening %s as mixer device...\n", mixer_name);

    if(stat(mixer_name, &statbuf) < 0) {
        fprintf(stderr, "Can't access %s\n", mixer_name);
        exit(1);
    }
    if(!(statbuf.st_mode & S_IFCHR)) {
        fprintf(stderr, "%s Is not a character device\n", mixer_name);
        exit(1);
    }
    if(((mixerdev = open(mixer_name, O_RDWR | O_NDELAY)) < 0) ||
       (fcntl(audiodev, F_SETFL, O_RDONLY) < 0)) {
        perror(audio_name);
        exit(1);
    }
#endif
}

#if defined (USE_SOLARIS_AUDIO)
void audioioctl(int req, audio_info_t *arg) {
#elif defined (USE_LINUX_AUDIO)
void audioioctl(int req, void *arg) {
#endif
    if(ioctl(audiodev, req, arg) < 0) {
        fprintf(stderr, "ioctl(audio, %d, ...) error on ", req);
        perror(audio_name);
        exit(2);
    }
}

#if defined (USE_LINUX_AUDIO)
void mixerioctl(int req, void *arg) {
    if(ioctl(mixerdev, req, arg) < 0) {
        fprintf(stderr, "ioctl(mixer, %d, ...) error on ", req);
        perror(audio_name);
        exit(2);
    }
}
#endif

void init_audio() {
#if defined (USE_SOLARIS_AUDIO)
    audio_info_t audio;

    AUDIO_INITINFO(&audio);
    audioioctl(AUDIO_GETINFO, &audio);
    monitor_gain = audio.monitor_gain;
    record_gain = audio.record.gain;

    AUDIO_INITINFO(&audio);
    audio.record.port = AUDIO_LINE_IN;
    audio.record.encoding = AUDIO_ENCODING_LINEAR;
    audio.record.sample_rate = sample_rate;
    audio.record.precision = 8;
    audio.record.channels = 1;
    audioioctl(AUDIO_SETINFO, &audio);
#elif defined (USE_LINUX_AUDIO)
    int iarg;
    struct mixer_info mi;
	
    mixerioctl(SOUND_MIXER_READ_DEVMASK, &iarg);
    if(iarg & (SOUND_MASK_RECLEV | SOUND_MASK_IGAIN | SOUND_MASK_LINE)) {
        if(iarg & SOUND_MASK_RECLEV) {
            mixer_reclev_n = SOUND_MIXER_RECLEV;
        } else if(iarg & SOUND_MASK_IGAIN) {
            mixer_reclev_n = SOUND_MIXER_IGAIN;
        } else if(iarg & SOUND_MASK_LINE) {
            mixer_reclev_n = SOUND_MIXER_LINE;
        }
        mixerioctl(MIXER_READ(mixer_reclev_n), &record_gain);
        record_gain &= 0xff;
        if(verbose) fprintf(stderr, "Recording level set by \"%s\" (%d)\n", devlabels[mixer_reclev_n], mixer_reclev_n);
    } else {
        fprintf(stderr, "Warning: can't set recording level on %s (%XH)\n", mixer_name, iarg);
        mixerioctl(SOUND_MIXER_INFO, &mi);
        fprintf(stderr, "Mixer %s: <<%s>> (MC=%d)\n", mi.id, mi.name, mi.modify_counter);
    }

    if(iarg & (SOUND_MASK_MONITOR | SOUND_MASK_IMIX | SOUND_MASK_LINE)) {
        if(iarg & SOUND_MASK_LINE) {
            mixer_monitor_n = SOUND_MIXER_LINE;
        } else if(iarg & SOUND_MASK_MONITOR) {
            mixer_monitor_n = SOUND_MIXER_MONITOR;
        } else if(iarg & SOUND_MASK_IMIX) {
            mixer_monitor_n = SOUND_MIXER_IMIX;
        }
        mixerioctl(MIXER_READ(mixer_monitor_n), &monitor_gain);
        monitor_gain &= 0xff;
        if(verbose) fprintf(stderr, "Monitoring device is \"%s\" (%d)\n", devlabels[mixer_monitor_n], mixer_monitor_n);
        if(mixer_reclev_n == mixer_monitor_n)
            fputs("Warning: the monitoring device is the same as the recording and will be ignored\n", stderr);
    } else {
        fprintf(stderr, "Warning: %s doesn't support recording monitor(s) (%XH, %d, %d)\n", mixer_name, iarg,
                SOUND_MIXER_MONITOR, SOUND_MIXER_IMIX);
        mixerioctl(SOUND_MIXER_INFO, &mi);
        fprintf(stderr, "Mixer %s: <<%s>> (MC=%d)\n", mi.id, mi.name, mi.modify_counter);
    }

    iarg = SOUND_MASK_LINE;
    mixerioctl(MIXER_WRITE(SOUND_MIXER_RECSRC), &iarg);
    if(iarg != SOUND_MASK_LINE) {
        fprintf(stderr, "%s doesn't support recording from line input (%XH, %XH)\n", mixer_name, iarg, SOUND_MASK_LINE);
        exit(2);
    }

    iarg = AFMT_U8;
    audioioctl(SNDCTL_DSP_SETFMT, &iarg);
    if(iarg != AFMT_U8) {
        fprintf(stderr, "%s doesn't support linear 8-bit encoding (AFMT_U8)\n", audio_name);
        exit(2);
    }
    iarg = 1; /* Set mono mode */
    audioioctl(SNDCTL_DSP_CHANNELS, &iarg);
    if (iarg != 1) {
        fprintf(stderr, "%s doesn't support mono mode\n", audio_name);
        exit(2);
    }
    iarg = sample_rate;
    audioioctl(SNDCTL_DSP_SPEED, &iarg);
    if (iarg != sample_rate) {
        fprintf(stderr, "Warning: %s doesn't support default sample rate of %d (set to %d)\n", audio_name, sample_rate, iarg);
    }

    audioioctl(SNDCTL_DSP_GETBLKSIZE, &iarg);
    if(verbose) fprintf(stderr, "Fragment size set to %d bytes\n", iarg);
#endif
}

int get_next_sample() {
    static unsigned char buff1sec[MAX_SAMPLE_RATE];
    static int index = MAX_SAMPLE_RATE;
    int j, nb;

    if(index >= sample_rate) {
        for(j = 0; j < sample_rate; j += nb) {
          if((nb = read(audiodev, &buff1sec[j], sample_rate - j)) < 0) {
            perror("get_next_sample -> read(audiodev)");
            exit(2);
          }
        }
        index = 0;
    }
    return buff1sec[index++] < edge_level ? 0 : 1;
}

void set_rec_gain(int val) {
#if defined (USE_SOLARIS_AUDIO)
    audio_info_t audio;

    AUDIO_INITINFO(&audio);
    record_gain =
    audio.record.gain = (AUDIO_MAX_GAIN - AUDIO_MIN_GAIN) * val / 100 + AUDIO_MIN_GAIN;
    audioioctl(AUDIO_SETINFO, &audio);
#elif defined (USE_LINUX_AUDIO)
    int iarg = val + (val << 8);

    if(mixer_reclev_n >= 0) {
        mixerioctl(MIXER_WRITE(mixer_reclev_n), &iarg);
        if(iarg != val + (val << 8)) {
            fprintf(stderr, "%s doesn't allow to set rec. gain %d (%XH)\n", mixer_name, val, iarg);
            exit(2);
        }
        record_gain = val;
    }
#endif
}

void set_monitor_gain(int val) {
#if defined (USE_SOLARIS_AUDIO)
    audio_info_t audio;

    AUDIO_INITINFO(&audio);
    monitor_gain =
    audio.monitor_gain = (AUDIO_MAX_GAIN - AUDIO_MIN_GAIN) * val / 100 + AUDIO_MIN_GAIN;
    audioioctl(AUDIO_SETINFO, &audio);
#elif defined (USE_LINUX_AUDIO)
    int iarg = val + (val << 8);
    if((mixer_monitor_n >= 0) && (mixer_reclev_n != mixer_monitor_n)) {
        mixerioctl(MIXER_WRITE(mixer_monitor_n), &iarg);
        if(iarg != val + (val << 8)) {
            fprintf(stderr, "%s doesn't allow to set monitor gain %d (%XH)\n", mixer_name, val, iarg);
            exit(2);
        }
        monitor_gain = val;
    }
#endif
}

void set_sample_rate(int val) {
#if defined (USE_SOLARIS_AUDIO)
    audio_info_t audio;

    AUDIO_INITINFO(&audio);
    sample_rate = audio.record.sample_rate = val;
    audioioctl(AUDIO_SETINFO, &audio);
#elif defined (USE_LINUX_AUDIO)
    int iarg = val;

    audioioctl(SNDCTL_DSP_SPEED, &iarg);
    if(iarg != val) {
        fprintf(stderr, "%s doesn't allow to set sample rate %dK\n", audio_name, val);
        exit(2);
    }
    sample_rate = val;
#endif
}

void closeall() {
    close(audiodev);
#if defined (USE_LINUX_AUDIO)
    close(mixerdev);
#endif
}
