/***************************************************************************
                          analyze.c  -  samples analyze
                             -------------------

    bkread - utility to read BK tapes via soundcard

    begin                : Sun Oct 28 2001
    copyright            : (C) 2001 by Alexander "las" Lunev
    email                : imbecyle@mail.ru
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "bkread.h"
#include "audio.h"
#include <stdio.h>
#include <stdlib.h>

int period0, min_0_period, max_0_period,
    period1, min_1_period, max_1_period,
    periodS, min_S_period, max_S_period,
    max_period;

int max_adaptations = 0;
int pattern_count = 03140;
int pattern_start = 1;
int adapt_tries = 0;

int read_pattern(int max) {
    int sample_count;
    int counter;
    static int last_sample = -1;

    for(counter = 1; last_sample != pattern_start; last_sample = get_next_sample())
        if(max && (++counter >= max)) return -1;

    counter = 0;

    for(sample_count = 1;
        last_sample == pattern_start;
        last_sample = get_next_sample()) {
        sample_count++;
        if(max && (++counter >= max)) return -1;
    }

    return last_pattern = sample_count;
}

int find_sync() {
    int cycle_length, pattern_repeats, pattern_length;
    int tune_pattern_count = pattern_count / 021,
        sync_pattern_count = pattern_count - tune_pattern_count;

    if(verbose)
        fprintf(stderr, "\nListening for header synchros (S=%d; T=%d)...\n",
                sync_pattern_count, tune_pattern_count);

    for(pattern_length = -1; pattern_length < 0;) {
        for(pattern_length = pattern_repeats = 0; pattern_repeats < sync_pattern_count; ) {
            cycle_length = read_pattern(0);
            if((cycle_length >= 4) && (abs(pattern_length - cycle_length) <= (pattern_length / 3))) {
                pattern_repeats++;
            } else {
                pattern_length = (cycle_length >= 4) ? cycle_length : 4;
                pattern_repeats = 0;
            }
        }

        if(verbose) fprintf(stderr, "%d synchros detected. Pattern length = %d.\n", pattern_repeats, pattern_length);

        for(pattern_length = pattern_repeats = 0; pattern_repeats < tune_pattern_count; ) {
            if((cycle_length = read_pattern(pattern_length * 3)) >= 3) {
                pattern_length = (cycle_length + pattern_length) / 2 +
                                 (cycle_length + pattern_length) % 2;
                pattern_repeats++;
            } else { pattern_length = -1; break; }
        }

        if(verbose) {
            if(pattern_length >= 0)
                fprintf(stderr, "Pattern length after %d tuning synchros = %d.\n", pattern_repeats, pattern_length);
            else
                fprintf(stderr, "Error while pattern tuning.\n");
        }
    }

    return pattern_length;
}

int analyze_pattern(int length) {
static char s[64];

    if((length >= min_0_period) && (length <= max_0_period)) {
        return PATTERN_ZERO;
    } else if((length >= min_1_period) && (length <= max_1_period)) {
        return PATTERN_ONE;
    } else if((length >= min_S_period) && (length <= max_S_period)) {
        return PATTERN_SYNC;
    } else {
        sprintf(s, "Unrecognized bit pattern (l=%d)", length);
        last_error = s;
        return PATTERN_ERR;
    }
}

void apply_new_periods(int p0) {
    period0 = p0;
    period1 = period0 * 2;
    periodS = period0 * 4;

    max_period = period0 * 20;
    min_0_period = period0 - period0 / 2;
    max_0_period = period0 + (period1 - period0) / 2;
    min_1_period = max_0_period + 1;
    max_1_period = period1 + (periodS - period1) / 2;
    min_S_period = max_1_period + 1;
    max_S_period = periodS + (periodS - period1) / 2;
}

void calculate_periods(int p0) {
    int i;

    apply_new_periods(p0);

    if(verbose) {
        fprintf(stderr, "Patterns lengths: 0 = %d; 1 = %d.\n", period0, period1);
        fprintf(stderr, "\nSignal Period Analyzer Line (2..%d):\n", max_S_period);
        for(i = 2; i <= max_S_period; i++)
            switch(analyze_pattern(i)) {
                case PATTERN_ZERO:
                    fprintf(stderr, "0");
                    break;
                case PATTERN_ONE:
                    fprintf(stderr, "1");
                    break;
                case PATTERN_SYNC:
                    fprintf(stderr, "S");
                    break;
                case PATTERN_ERR:
                    fprintf(stderr, "-");
                    break;
                default:
                    fprintf(stderr, "?");
                    break;
            }
        fprintf(stderr, "\n");
        for(i = 2; i <= max_S_period; i++)
            fprintf(stderr, "-");
        fprintf(stderr, "\n");
        for(i = 2; i <= max_S_period; i++)
            fprintf(stderr, "%d", i / 10);
        fprintf(stderr, "\n");
        for(i = 2; i <= max_S_period; i++)
            fprintf(stderr, "%d", i % 10);
        fprintf(stderr, "\n\n");
    }
}

int try_adapt(int pattern, int to) {
    int i;
    int p0, oldp0 = period0;
    int dev = (period0 > 3) ? period0 / 3 : 1;

    if(verbose) fprintf(stderr, "\n*** Trying to adapt pattern length %d to %d ***\n", pattern, to);
    if(max_adaptations && (++adapt_tries > max_adaptations)) {
        if(verbose) fprintf(stderr, "\n*** Too many adaptations ***\n");
        return -1;
    }

    for(i = 1; i <= dev; i++) {
        for(p0 = dev - i; p0 <= dev + i; p0 += 2 * i) {
            apply_new_periods(p0);
            if(analyze_pattern(pattern) == to) {
                if(verbose) fprintf(stderr, "\n*** New bit rate adaptation successfull ***\n");
                calculate_periods(p0);
                return to;
            }
        }
    }
    apply_new_periods(oldp0);
    return -1;
}

int read_bit() {
    int prev;

    switch(analyze_pattern(read_pattern(max_period))) {
        case PATTERN_ZERO:
            prev = PATTERN_ZERO;
            break;
        case PATTERN_ONE:
            prev = PATTERN_ONE;
            break;
        case PATTERN_SYNC:
            if(try_adapt(last_pattern, PATTERN_ONE) > 0) {
                prev = PATTERN_ONE;
                break;
            }
            last_error = "PATTERN_SYNC at the start of bit";
            return -1;
        case PATTERN_ERR:
            if(try_adapt(last_pattern, PATTERN_ZERO) > 0) {
                prev = PATTERN_ZERO;
                break;
            }
            if(verbose) fprintf(stderr, "PATTERN_ERR in bit - %s.\n", last_error);
            last_error = "PATTERN_ERR at the start of bit";
            return -1;
        default:
            fprintf(stderr, "Unexpected internal error (2).\n");
            exit(2);
    }
    switch(analyze_pattern(read_pattern(max_period))) {
        case PATTERN_ZERO:
            return (prev == PATTERN_ZERO) ? 0 : 1;
        case PATTERN_ONE:
            if(try_adapt(last_pattern, PATTERN_ZERO) == PATTERN_ZERO)
                return (prev == PATTERN_ZERO) ? 0 : 1;
            fprintf(stderr, "Bit synchro = PATTERN_ONE (pl=%d, p=%d).\n", last_pattern, prev);
            last_error = "Bit synchro not PATTERN_ZERO";
            return -1;
        case PATTERN_SYNC:
            last_error = "PATTERN_SYNC at the end of bit";
            return -1;
        case PATTERN_ERR:
            if(try_adapt(last_pattern, PATTERN_ZERO) == PATTERN_ZERO)
                return (prev == PATTERN_ZERO) ? 0 : 1;
            if(verbose) fprintf(stderr, "PATTERN_ERR in bit - %s.\n", last_error);
            last_error = "PATTERN_ERR at the end of bit";
            return -1;
        default:
            fprintf(stderr, "Unexpected internal error (3).\n");
            exit(2);
    }
}

void detect_preamble() {
    int correct_header;
    int i;

    for(correct_header = 0; !correct_header; ) {
        adapt_tries = 0;
        calculate_periods(find_sync());

        for(i = 1; i;) {                               // Skip sync zeros
            switch(analyze_pattern(read_pattern(max_period))) {
                case PATTERN_ZERO:
                    break;
                case PATTERN_ONE:
                    if(try_adapt(last_pattern, PATTERN_SYNC) == PATTERN_SYNC) {
                        i = 0;
                        correct_header = 1;
                        break;
                    }
                    if(try_adapt(last_pattern, PATTERN_ZERO) < 0) break;
                    if(verbose) fprintf(stderr, "Unexpected \"1\" bit synchros (pl=%d).\n", last_pattern);
                    break;
                case PATTERN_SYNC:
                    i = 0;
                    correct_header = 1;
                    break;
                case PATTERN_ERR:
                    if(try_adapt(last_pattern, PATTERN_SYNC) == PATTERN_SYNC) {
                        i = 0;
                        correct_header = 1;
                        break;
                    }
                    if(verbose) fprintf(stderr, "PATTERN_ERR in synchros - %s.\n", last_error);
                    i = 0;
                    break;
                default:
                    fprintf(stderr, "Unexpected internal error (1).\n");
                    exit(2);
            }
        }

        if(correct_header && (i = read_bit()) != 1) {
            if(verbose) {
                fprintf(stderr, "Synchro sequency doesn't finish by \"1\" bit (%d)", i);
                if(i < 0) fprintf(stderr, " - %s.\n", last_error);
                else fprintf(stderr, ".\n");
            }
            correct_header = 0;
        }
    }
}

int detect_marker() {
    int i;

    for(i = 0; i < 8; i++) {
        if(analyze_pattern(read_pattern(max_period)) != PATTERN_ZERO &&
           (try_adapt(last_pattern, PATTERN_ZERO) < 0)) {
            if(verbose) fprintf(stderr, "Invalid pattern in the marker synchros.\n");
            last_error = "Nonzero pattern in the marker body";
            return -1;
        }
    }

    if((i = analyze_pattern(read_pattern(max_period))) != PATTERN_SYNC &&
       (try_adapt(last_pattern, PATTERN_SYNC) < 0)) {
        if(i == PATTERN_ERR) {
            if(verbose) fprintf(stderr, "PATTERN_ERR detected pl=%d.\n", last_pattern);
            last_error = "Found PATTERN_ERR at the end of marker";
            return -1;
        }
        if(verbose)
            fprintf(stderr, "Marker doesn't end by PATTERN_SYNC (pl=%d).\n", last_pattern);
        last_error = "Illegal pattern at the end of marker";
        return -1;
    }

    if((i = read_bit()) != 1) {
        if(verbose) {
            fprintf(stderr, "Marker doesn't finish by \"1\" bit (%d)\n", i);
            if(i < 0) fprintf(stderr, " - %s.\n", last_error);
            else fprintf(stderr, ".\n");
        }
        last_error = "No \"1\" bit at the end of marker";
        return -1;
    }

    return 1;
}

int read_byte(unsigned char *byte) {
    int i;

    *byte = 0;
    for(i = 0; i < 8; i++) {
        switch(read_bit()) {
            case 1:
                *byte |= 1 << i;
            case 0:
                break;
            default:
                if(verbose) fprintf(stderr, "Error reading byte - %s; bit%d; byte=%o (oct).\n", last_error, i, *byte);
                last_error = "Illegal bit pattern";
                return -1;
        }
    }
    return *byte;
}
