--
--

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity bkv1610 is
port(
	-- VGA
--	vgar : out std_logic_vector(4 downto 0);
--	vgag : out std_logic_vector(5 downto 0);
--	vgab : out std_logic_vector(4 downto 0);
--	vgah : out std_logic;
--	vgav : out std_logic;
	-- HDMI
	HDMI_D0		: out std_logic;
	HDMI_D0N		: out std_logic;
	HDMI_D1		: out std_logic;
	HDMI_D1N		: out std_logic;
	HDMI_D2		: out std_logic;
	HDMI_D2N		: out std_logic;
	HDMI_CLK		: out std_logic;
	HDMI_CLKN	: out std_logic;

	basic : in std_logic;

-- serial port
	tx : out std_logic;
	rx : in std_logic;
-- ps2 keyboard
	ps2k_c : in std_logic;
	ps2k_d : in std_logic;
-- debug
	TEST	: out std_logic_vector(7 downto 0);
---	test_bus_addr	: out std_logic_vector(17 downto 0);
---- SDRAM
--	DRAM_DQ		: inout std_logic_vector(15 downto 0);
--	DRAM_A		: out std_logic_vector(11 downto 0);
--	DRAM_CLK	: out std_logic;
--	DRAM_NCAS	: out std_logic;
--	DRAM_NRAS	: out std_logic;
--	DRAM_BA		: out std_logic_vector(1 downto 0);
--	DRAM_NWE	: out std_logic;
--	DRAM_DQM	: out std_logic_vector(1 downto 0);
	-- Audio (Delta-sigma)
	DAC_OUT_L	: out std_logic;
	DAC_OUT_R	: out std_logic;
-- clock & reset
	clk100mhz : in std_logic;
	nreset : in std_logic
   );
end bkv1610;

architecture implementation of bkv1610 is

--COMPONENT fiftyfivenm_adcblock_top_wrapper is
--GENERIC (analog_input_pin_mask : INTEGER;
--			clkdiv : INTEGER;
--			device_partname_fivechar_prefix : STRING;
--			hard_pwd : INTEGER;
--			is_this_first_or_second_adc : INTEGER;
--			prescalar : INTEGER;
--			refsel : INTEGER;
--			tsclkdiv : INTEGER;
--			tsclksel : INTEGER
--			);
--	PORT(soc : IN STD_LOGIC;
--		 usr_pwd : IN STD_LOGIC;
--		 tsen : IN STD_LOGIC;
--		 clkin_from_pll_c0 : IN STD_LOGIC;
--		 chsel : IN STD_LOGIC_VECTOR(4 DOWNTO 0);
--		 eoc : OUT STD_LOGIC;
--		 clkout_adccore : OUT STD_LOGIC;
--		 dout : OUT STD_LOGIC_VECTOR(11 DOWNTO 0)
--	);
--END COMPONENT;

component vm1_reset is
   port(
	clk : in std_logic;--- 
	reset : in std_logic;--- 
	dclo : out std_logic;---
	aclo : out std_logic---
   );
end component;

component vm1_qbus is
   port(
   pin_clk_p : in std_logic;---        // processor clock
   pin_clk_n : in std_logic;---        // processor clock 180 degree
   pin_ena : in std_logic;---          // processor clock enable
   pin_pa : in std_logic_vector(1 downto 0);---           // processor number
   pin_init_in : in std_logic;---      // peripheral reset input
   pin_init_out : out std_logic;---     // peripheral reset output
 
   pin_dclo : in std_logic;---         // processor reset
   pin_aclo : in std_logic;---         // power fail notoficaton
   pin_irq : in std_logic_vector(3 downto 1);---          // radial interrupt requests
   pin_virq : in std_logic;---         // vectored interrupt request

   pin_ad_in : in std_logic_vector(15 downto 0);---        // data bus input
   pin_ad_out : out std_logic_vector(15 downto 0);---       // address/data bus output
   pin_ad_ena : out std_logic;---       // address/data bus enable

   pin_dout_in : in std_logic;---      // data output strobe input
   pin_dout_out : out std_logic;---     // data output strobe output
   pin_din_in : in std_logic;---       // data input strobe input
   pin_din_out : out std_logic;---      // data input strobe output
   pin_wtbt : out std_logic;---         // write/byte status
   pin_ctrl_ena : out std_logic;---     // enable control outputs

   pin_sync_in : in std_logic;---      // address strobe input
   pin_sync_out : out std_logic;---     // address strobe output
   pin_sync_ena : out std_logic;---     // address strobe enable

   pin_rply_in : in std_logic;---      // transaction reply input
   pin_rply_out : out std_logic;---     // transaction reply output

   pin_dmr_in : in std_logic;---       // bus request shared line
   pin_dmr_out : out std_logic;---      //

   pin_sack_in : in std_logic;---      // bus acknowlegement
   pin_sack_out : out std_logic;---     // bus acknowlegement

   pin_dmgi : in std_logic;---         // bus granted input
   pin_dmgo : out std_logic;---         // bus granted output
   pin_iako : out std_logic;---         // interrupt vector input
   pin_sp : in std_logic;---           // peripheral timer input
   pin_sel : out std_logic_vector(2 downto 1);---          // register select outputs
   pin_bsy : out std_logic---           // bus busy flag
   );
end component;

--component sdram is
--port (
--	CLK_I	: in std_logic;
--	PRES_I	: in std_logic;
--	-- Memory port
--	ADDR_I	: in std_logic_vector(23 downto 0);
--	DATA_I	: in std_logic_vector(15 downto 0);
--	DATA_O	: out std_logic_vector(15 downto 0);
--	WR_I	: in std_logic;
--	BE_I	: in std_logic_vector(1 downto 0);
--	RD_I	: in std_logic;
--	RFSH_I	: in std_logic;
--	IDLE_O	: out std_logic;
--	-- SDRAM Pin
--	CLK_O	: out std_logic;
--	RAS_O	: out std_logic;
--	CAS_O	: out std_logic;
--	WE_O	: out std_logic;
--	DQM_O	: out std_logic_vector(1 downto 0);
--	BA_O	: out std_logic_vector(1 downto 0);
--	MA_O	: out std_logic_vector(11 downto 0);
--	DQ_IO	: inout std_logic_vector(15 downto 0) );
--end component;

component dp0ram16x8k is
   port(
		address_a		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		address_b		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		byteena_a		: IN STD_LOGIC_VECTOR (1 DOWNTO 0) :=  (OTHERS => '1');
		clock		: IN STD_LOGIC  := '1';
		data_a		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		data_b		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		wren_a		: IN STD_LOGIC  := '0';
		wren_b		: IN STD_LOGIC  := '0';
		q_a		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0);
		q_b		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
	);
end component;

component dp1ram16x8k is
   port(
		address_a		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		address_b		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		byteena_a		: IN STD_LOGIC_VECTOR (1 DOWNTO 0) :=  (OTHERS => '1');
		clock		: IN STD_LOGIC  := '1';
		data_a		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		data_b		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		wren_a		: IN STD_LOGIC  := '0';
		wren_b		: IN STD_LOGIC  := '0';
		q_a		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0);
		q_b		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
	);
end component;

component dp2ram16x8k is
   port(
		address_a		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		address_b		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		byteena_a		: IN STD_LOGIC_VECTOR (1 DOWNTO 0) :=  (OTHERS => '1');
		clock		: IN STD_LOGIC  := '1';
		data_a		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		data_b		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		wren_a		: IN STD_LOGIC  := '0';
		wren_b		: IN STD_LOGIC  := '0';
		q_a		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0);
		q_b		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
	);
end component;

component dp3ram16x8k is
   port(
		address_a		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		address_b		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		byteena_a		: IN STD_LOGIC_VECTOR (1 DOWNTO 0) :=  (OTHERS => '1');
		clock		: IN STD_LOGIC  := '1';
		data_a		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		data_b		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		wren_a		: IN STD_LOGIC  := '0';
		wren_b		: IN STD_LOGIC  := '0';
		q_a		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0);
		q_b		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
	);
end component;

component dp4ram16x8k is
   port(
		address_a		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		address_b		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		byteena_a		: IN STD_LOGIC_VECTOR (1 DOWNTO 0) :=  (OTHERS => '1');
		clock		: IN STD_LOGIC  := '1';
		data_a		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		data_b		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		wren_a		: IN STD_LOGIC  := '0';
		wren_b		: IN STD_LOGIC  := '0';
		q_a		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0);
		q_b		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
	);
end component;

component dp5ram16x8k is
   port(
		address_a		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		address_b		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		byteena_a		: IN STD_LOGIC_VECTOR (1 DOWNTO 0) :=  (OTHERS => '1');
		clock		: IN STD_LOGIC  := '1';
		data_a		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		data_b		: IN STD_LOGIC_VECTOR (15 DOWNTO 0);
		wren_a		: IN STD_LOGIC  := '0';
		wren_b		: IN STD_LOGIC  := '0';
		q_a		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0);
		q_b		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
	);
end component;

component rom016x4k is
   port(
		address		: IN STD_LOGIC_VECTOR (11 DOWNTO 0);
		clock		: IN STD_LOGIC  := '1';
		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
   );
end component;

component rom116x4k is
   port(
		address		: IN STD_LOGIC_VECTOR (11 DOWNTO 0);
		clock		: IN STD_LOGIC  := '1';
		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
   );
end component;

component rom316x4k is
   port(
		address		: IN STD_LOGIC_VECTOR (11 DOWNTO 0);
		clock		: IN STD_LOGIC  := '1';
		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
   );
end component;

--component rom1basic is
--   port(
--		address		: IN STD_LOGIC_VECTOR (11 DOWNTO 0);
--		clock		: IN STD_LOGIC  := '1';
--		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
--   );
--end component;

component rom0216x4k is
   port(
		address		: IN STD_LOGIC_VECTOR (11 DOWNTO 0);
		clock		: IN STD_LOGIC  := '1';
		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
   );
end component;

component rom1216x4k is
   port(
		address		: IN STD_LOGIC_VECTOR (11 DOWNTO 0);
		clock		: IN STD_LOGIC  := '1';
		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
   );
end component;

component rom2216x4k is
   port(
		address		: IN STD_LOGIC_VECTOR (11 DOWNTO 0);
		clock		: IN STD_LOGIC  := '1';
		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
   );
end component;

--component rom2basic is
--   port(
--		address		: IN STD_LOGIC_VECTOR (11 DOWNTO 0);
--		clock		: IN STD_LOGIC  := '1';
--		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
--   );
--end component;

component rom0316x4k is
   port(
		address		: IN STD_LOGIC_VECTOR (11 DOWNTO 0);
		clock		: IN STD_LOGIC  := '1';
		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
   );
end component;

component rom1316x4k is
   port(
		address		: IN STD_LOGIC_VECTOR (11 DOWNTO 0);
		clock		: IN STD_LOGIC  := '1';
		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
   );
end component;

component rom2316x4k is
   port(
		address		: IN STD_LOGIC_VECTOR (11 DOWNTO 0);
		clock		: IN STD_LOGIC  := '1';
		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
   );
end component;

--component rom3basic is
--   port(
--		address		: IN STD_LOGIC_VECTOR (11 DOWNTO 0);
--		clock		: IN STD_LOGIC  := '1';
--		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
--   );
--end component;

component bk1610vga is
	port(
		pres   : in std_logic; --- active high
		aclk   : in std_logic; --- 100 MHz
		bk1vdata: in std_logic_vector(15 downto 0); --- BK 2 port VRAM data
		bk3vdata: in std_logic_vector(15 downto 0); --- BK 2 port VRAM data
		bk5vdata: in std_logic_vector(15 downto 0); --- BK 2 port VRAM data
		border : in std_logic_vector(2 downto 0); --- colour of border
		icolor : in std_logic; --- White/Color
		isimd : in std_logic; --- SIMD
		ipalnum : in std_logic_vector(3 downto 0); --- Palette number
---		vgaport : in std_logic_vector(15 downto 0); --- vga port
		bkvaddr: out std_logic_vector(15 downto 0); --- address of BK VRAM
		blank  : out std_logic; --- active high
		vgar   : out std_logic_vector(7 downto 0); --- R
		vgag   : out std_logic_vector(7 downto 0); --- G
		vgab   : out std_logic_vector(7 downto 0); --- B 
		vsync  : out std_logic; --- active low
		hsync  : out std_logic; --- active low

		clken : in STD_LOGIC;
		rd : in std_logic;
		wr : in std_logic;
		addr_match : out std_logic;
		base_addr : in std_logic_vector(17 downto 0); --- o177664
		addr  : in std_logic_vector(17 downto 0);
		din : in STD_LOGIC_VECTOR(15 downto 0);
		dout : out STD_LOGIC_VECTOR(15 downto 0);

		oInt   : out std_logic --- active high
	);
end component;

component bkkbd is
port(
	pres : in STD_LOGIC;
	clk : in STD_LOGIC;
	clken : in STD_LOGIC;
	br : out std_logic;
	bg : in std_logic;
	int_vector : out std_logic_vector(8 downto 0);
	rd : in std_logic;
	wr : in std_logic;
	dw8 : in std_logic;
	keyupdn : out std_logic;
	kbdreset : out std_logic;
	kbdcolor : out std_logic;
	kbdsimd : out std_logic;
	kbdstop : out std_logic;
	addr_match : out std_logic;
	base_addr : in std_logic_vector(17 downto 0); --- o177660
	addr  : in std_logic_vector(17 downto 0);
	din : in STD_LOGIC_VECTOR(15 downto 0);
	kbci : in STD_LOGIC;
	kbdi : in STD_LOGIC;
	opalnum : out STD_LOGIC_VECTOR(3 downto 0);
	dout : out STD_LOGIC_VECTOR(15 downto 0);
	TEST : out STD_LOGIC_VECTOR(7 downto 0)
	);
end component;

component bkvpll is
   port(
		inclk0		: IN STD_LOGIC  := '0';
		c0		: OUT STD_LOGIC ;
		c1		: OUT STD_LOGIC ;
		c2		: OUT STD_LOGIC ;
		c3		: OUT STD_LOGIC ;
		c4		: OUT STD_LOGIC ;
		locked		: OUT STD_LOGIC 
   );
end component;

-- HDMI
component hdmi is
port(
	CLK_DVI_I	: in std_logic;
	CLK_PIXEL_I	: in std_logic;
	R_I		: in std_logic_vector(7 downto 0);
	G_I		: in std_logic_vector(7 downto 0);
	B_I		: in std_logic_vector(7 downto 0);
	BLANK_I		: in std_logic;
	HSYNC_I		: in std_logic;
	VSYNC_I		: in std_logic;
	HDMI_D0_O	: out std_logic;
	HDMI_D0N_O	: out std_logic;
	HDMI_D1_O	: out std_logic;
	HDMI_D1N_O	: out std_logic;
	HDMI_D2_O	: out std_logic;
	HDMI_D2N_O	: out std_logic;
	HDMI_CLK_O	: out std_logic;
	HDMI_CLKN_O	: out std_logic
	);
end component;

---constant init_r7 : std_logic_vector(15 downto 0) := x"8000";         -- start address after reset = o'100000'
signal init_r7: std_logic_vector(15 downto 0);
constant init_psw : std_logic_vector(15 downto 0) := x"00e0";        -- initial psw for kernel mode, primary register set, priority 7

signal cpu_addr : std_logic_vector(15 downto 0);
signal cpu0_datain : std_logic_vector(15 downto 0);
signal cpu1_datain : std_logic_vector(15 downto 0);
signal cpu2_datain : std_logic_vector(15 downto 0);
signal cpu0_dataout : std_logic_vector(15 downto 0);
signal cpu1_dataout : std_logic_vector(15 downto 0);
signal cpu2_dataout : std_logic_vector(15 downto 0);
signal dp0_datain : std_logic_vector(15 downto 0);
signal dp1_datain : std_logic_vector(15 downto 0);
signal dp2_datain : std_logic_vector(15 downto 0);
signal cpu_wr : std_logic;
signal cpu_rd : std_logic;
signal cpu_be : std_logic_vector(1 downto 0);
signal cpu_psw : std_logic_vector(15 downto 0);
signal cpu_psw_in : std_logic_vector(15 downto 0);
signal cpu_psw_we_even : std_logic;
signal cpu_psw_we_odd : std_logic;
signal cpu_pir_in : std_logic_vector(15 downto 0);
signal cpu_dw8 : std_logic;
---signal cpu_cp : std_logic;
signal cpu_id : std_logic;
signal cpu_init : std_logic;

signal cpu_irqu : std_logic;
signal cpu_iako : std_logic;
signal cpu_int_vector : std_logic_vector(8 downto 0);

signal bus_addr : std_logic_vector(17 downto 0);
---signal bus_dati : std_logic_vector(15 downto 0);
---signal cpu0_dataout : std_logic_vector(15 downto 0);
---signal cpu_wr : std_logic;
---signal cpu_dw8 : std_logic;

signal ps2_addr_match : std_logic;
signal ps2_dataout : std_logic_vector(15 downto 0);
signal keyupdn : std_logic;
signal kbdreset : std_logic;

---signal ram_addr_match : std_logic;
signal ram_dati : std_logic_vector(15 downto 0);

signal svgar : std_logic_vector(7 downto 0);
signal svgag : std_logic_vector(7 downto 0);
signal svgab : std_logic_vector(7 downto 0);
signal svgah : std_logic;
signal svgav : std_logic;
signal svgablank : std_logic;
signal vgacr_addr_match : std_logic;

signal sysclk : std_logic;
signal cpuclkp : std_logic;
signal cpuclkn : std_logic;
signal cpuclken : std_logic;
signal vgaclken : std_logic;
signal hdmiclk : std_logic;
signal vgaclk : std_logic;
signal locked : std_logic;
signal reset : std_logic;

signal sclkcnt : std_logic_vector(4 downto 0);
signal swcnt : std_logic_vector(6 downto 0);
signal vga_addr : std_logic_vector(15 downto 0);
signal sborder : std_logic_vector(2 downto 0);
---signal sdram_addr : std_logic_vector(23 downto 0);
signal sd0ram_dataout : std_logic_vector(15 downto 0);
signal sd2ram_dataout : std_logic_vector(15 downto 0);
signal sd4ram_dataout : std_logic_vector(15 downto 0);
signal dp1ram_dataout : std_logic_vector(15 downto 0);
signal dp3ram_dataout : std_logic_vector(15 downto 0);
signal dp5ram_dataout : std_logic_vector(15 downto 0);
signal dp1vga_dataout : std_logic_vector(15 downto 0);
signal dp3vga_dataout : std_logic_vector(15 downto 0);
signal dp5vga_dataout : std_logic_vector(15 downto 0);
signal rom0_dataout : std_logic_vector(15 downto 0);
signal rom1_dataout : std_logic_vector(15 downto 0);
signal rom3_dataout : std_logic_vector(15 downto 0);
signal rom02_dataout : std_logic_vector(15 downto 0);
signal rom12_dataout : std_logic_vector(15 downto 0);
signal rom22_dataout : std_logic_vector(15 downto 0);
signal rom03_dataout : std_logic_vector(15 downto 0);
signal rom13_dataout : std_logic_vector(15 downto 0);
signal rom23_dataout : std_logic_vector(15 downto 0);
signal rom303_dataout : std_logic_vector(15 downto 0);
signal rom313_dataout : std_logic_vector(15 downto 0);
signal rom323_dataout : std_logic_vector(15 downto 0);
---signal basic1_dataout : std_logic_vector(15 downto 0);
---signal basic2_dataout : std_logic_vector(15 downto 0);
---signal basic3_dataout : std_logic_vector(15 downto 0);
signal sysregs_dataout : std_logic_vector(15 downto 0);
signal sdram_wr : std_logic;
---signal sdram_rd : std_logic;
signal dpram_wr : std_logic;

signal port_664_reg : std_logic_vector(15 downto 0);
signal port_700_reg : std_logic_vector(15 downto 0);
signal port_704_reg : std_logic_vector(15 downto 0);
signal port_i702_reg : std_logic_vector(15 downto 0);
signal port_o702_reg : std_logic_vector(15 downto 0);
signal port_i714_reg : std_logic_vector(15 downto 0);
signal port_o714_reg : std_logic_vector(15 downto 0);
signal port_i716_reg : std_logic_vector(15 downto 0);
signal port_o716_reg : std_logic_vector(15 downto 0);

-- HDMI
signal sHDMI_D0	: std_logic;
signal sHDMI_D0N	: std_logic;
signal sHDMI_D1	: std_logic;
signal sHDMI_D1N	: std_logic;
signal sHDMI_D2	: std_logic;
signal sHDMI_D2N	: std_logic;
signal sHDMI_CLK	: std_logic;
signal sHDMI_CLKN	: std_logic;

signal sbasic		: std_logic;

-- Tape input from ADC
---signal ssgclk		: std_logic;
signal tape_in		: std_logic;
signal adc_d		: std_logic_vector(11 downto 0);

--- White/Color
signal scolor		: std_logic;
signal kbdcolor	: std_logic;
signal ssimd		: std_logic;
signal kbdsimd		: std_logic;

signal skbdstop		: std_logic;
signal skbd1stop		: std_logic;
signal skbd2stop		: std_logic;
signal sintstop		: std_logic;

signal svcolor		: std_logic;
signal svsimd		: std_logic;

signal spalnum		: std_logic_vector(3 downto 0);

signal sTEST		: std_logic_vector(7 downto 0);

signal cpu_dclo	: std_logic;
signal cpu_aclo	: std_logic;
signal cpu0_ad_in		: std_logic_vector(15 downto 0);
signal cpu0_ad_out	: std_logic_vector(15 downto 0);
signal cpu0_ad_ena	: std_logic;
signal cpu1_ad_in		: std_logic_vector(15 downto 0);
signal cpu1_ad_out	: std_logic_vector(15 downto 0);
signal cpu1_ad_ena	: std_logic;
signal cpu2_ad_in		: std_logic_vector(15 downto 0);
signal cpu2_ad_out	: std_logic_vector(15 downto 0);
signal cpu2_ad_ena	: std_logic;
signal cpu_sync	: std_logic;
signal cpu_dsync	: std_logic;
signal cpu_sel		: std_logic_vector(2 downto 1);
signal cpu_rply	: std_logic;
signal srply	: std_logic;
signal sdcpu_rd : std_logic;

signal sirq : std_logic_vector(2 downto 0);

begin

---tape_in <= '1' when adc_d > 96 else '0' when adc_d < 32;
---tape_in <= '1' when adc_d > 2047 else '0';
tape_in <= '1' when adc_d > 127 else '0' when adc_d < 32;

---sirq <= "00" & skbdstop;---Debug
sirq <= "00" & sintstop;

-- ADC
--U0: fiftyfivenm_adcblock_top_wrapper
--GENERIC MAP(analog_input_pin_mask => 0,
--			clkdiv => 1, --- 2 MHz
--			device_partname_fivechar_prefix => "10M50",
--			hard_pwd => 0,
--			is_this_first_or_second_adc => 1,
--			prescalar => 0,
--			refsel => 0,
--			tsclkdiv => 0,
--			tsclksel => 0
--			)
--PORT MAP(soc => '1',
--		 usr_pwd => '0',
--		 tsen => '1',
--		 clkin_from_pll_c0 => ssgclk,
--		 chsel => "00000",
--		 dout => adc_d);

pll0: bkvpll port map(
	inclk0 => clk100mhz,
	c0	=> hdmiclk, --- 125 MHz
	c1	=> sysclk, --- 100 MHz
	c2	=> vgaclk, --- 25 MHz
	c3	=> cpuclkp, --- 2,5 MHz
	c4	=> cpuclkn, --- 2,5 MHz
	locked => locked
	);

rst0: vm1_reset port map(
	clk => cpuclkp,
	reset => reset,
	dclo => cpu_dclo,
	aclo => cpu_aclo
	);

cpu0: vm1_qbus port map(
   pin_clk_p => cpuclkp,
   pin_clk_n => cpuclkn,
   pin_ena => '1', --- timer enable
   pin_pa => "00",
   pin_init_in => cpu_init,
   pin_init_out => cpu_init,

   pin_dclo => cpu_dclo,
   pin_aclo => cpu_aclo,
   pin_irq => sirq,
   pin_virq => cpu_irqu,

   pin_ad_in => cpu0_ad_in,
   pin_ad_out => cpu0_ad_out,
   pin_ad_ena => cpu0_ad_ena,

   pin_dout_in => cpu_wr,
   pin_dout_out => cpu_wr,
   pin_din_in => cpu_rd,
   pin_din_out => cpu_rd,
   pin_wtbt => cpu_dw8,
   pin_ctrl_ena => open,

   pin_sync_in => cpu_sync,
   pin_sync_out => cpu_sync,
   pin_sync_ena => open,

   pin_rply_in => srply,
   pin_rply_out => cpu_rply,

   pin_dmr_in => '0',
   pin_dmr_out => open,

   pin_sack_in => '0',
   pin_sack_out => open,

   pin_dmgi => '0',
   pin_dmgo => open,
   pin_iako => cpu_iako,
   pin_sp => '0',
   pin_sel => cpu_sel,
   pin_bsy => open
	);

cpu0_ad_in <= cpu0_ad_out when cpu0_ad_ena = '1' else cpu0_datain;

mk_sdcpu_rd:
process (cpuclkp,cpu_rd,cpu_wr)
begin
	if cpuclkp'event and cpuclkp = '1' then
		sdcpu_rd <= cpu_rd or cpu_wr after 1 ns;
	end if;
end process;

srply <= cpu_rply or sdcpu_rd;
	
cpu1: vm1_qbus port map(
   pin_clk_p => cpuclkp,
   pin_clk_n => cpuclkn,
   pin_ena => '1', --- timer enable
   pin_pa => "00",
   pin_init_in => cpu_init,
   pin_init_out => open,

   pin_dclo => cpu_dclo,
   pin_aclo => cpu_aclo,
   pin_irq => sirq,
   pin_virq => cpu_irqu,

   pin_ad_in => cpu1_ad_in,
   pin_ad_out => cpu1_ad_out,
   pin_ad_ena => cpu1_ad_ena,

   pin_dout_in => cpu_wr,
   pin_dout_out => open,
   pin_din_in => cpu_rd,
   pin_din_out => open,
   pin_wtbt => open,
   pin_ctrl_ena => open,

   pin_sync_in => cpu_sync,
   pin_sync_out => open,
   pin_sync_ena => open,

   pin_rply_in => srply,
   pin_rply_out => open,

   pin_dmr_in => '0',
   pin_dmr_out => open,

   pin_sack_in => '0',
   pin_sack_out => open,

   pin_dmgi => '0',
   pin_dmgo => open,
   pin_iako => open,
   pin_sp => '0',
   pin_sel => open,
   pin_bsy => open
	);

cpu1_ad_in <= cpu1_ad_out when cpu1_ad_ena = '1' else cpu1_datain;

cpu2: vm1_qbus port map(
   pin_clk_p => cpuclkp,
   pin_clk_n => cpuclkn,
   pin_ena => '1', --- timer enable
   pin_pa => "00",
   pin_init_in => cpu_init,
   pin_init_out => open,

   pin_dclo => cpu_dclo,
   pin_aclo => cpu_aclo,
   pin_irq => sirq,
   pin_virq => cpu_irqu,

   pin_ad_in => cpu2_ad_in,
   pin_ad_out => cpu2_ad_out,
   pin_ad_ena => cpu2_ad_ena,

   pin_dout_in => cpu_wr,
   pin_dout_out => open,
   pin_din_in => cpu_rd,
   pin_din_out => open,
   pin_wtbt => open,
   pin_ctrl_ena => open,

   pin_sync_in => cpu_sync,
   pin_sync_out => open,
   pin_sync_ena => open,

   pin_rply_in => srply,
   pin_rply_out => open,

   pin_dmr_in => '0',
   pin_dmr_out => open,

   pin_sack_in => '0',
   pin_sack_out => open,

   pin_dmgi => '0',
   pin_dmgo => open,
   pin_iako => open,
   pin_sp => '0',
   pin_sel => open,
   pin_bsy => open
	);

cpu2_ad_in <= cpu2_ad_out when cpu2_ad_ena = '1' else cpu2_datain;

init_r7 <= port_o716_reg(15 downto 8) & x"00";
cpu_be(0) <= (not cpu_dw8 and cpu_wr) or (cpu_wr and cpu_dw8 and not cpu_addr(0));
cpu_be(1) <= (not cpu_dw8 and cpu_wr) or (cpu_wr and cpu_dw8 and cpu_addr(0));
dp0_datain <= cpu0_dataout;
dp1_datain <= cpu1_dataout;
dp2_datain <= cpu2_dataout;
---sdram_addr <= "00000000" & cpu_addr(15 downto 0);
sdram_wr <= cpu_wr when cpu_addr(15 downto 14) = "00" else '0';
---sdram_rd <= cpu_rd when cpu_addr(15 downto 14) = "00" else '0';
dpram_wr <= cpu_wr when cpu_addr(15 downto 14) = "01" else '0';

cpu0_dataout <= cpu0_ad_out;
cpu1_dataout <= cpu1_ad_out;
cpu2_dataout <= cpu2_ad_out;

rom303_dataout <= rom03_dataout when port_o702_reg(3) = '1' else rom3_dataout;
rom313_dataout <= rom13_dataout when port_o702_reg(3) = '1' else rom3_dataout;
rom323_dataout <= rom23_dataout when port_o702_reg(3) = '1' else rom3_dataout;
--rom303_dataout <= rom03_dataout;
--rom313_dataout <= rom13_dataout;
--rom323_dataout <= rom23_dataout;

---cpu0_datain <= sd0ram_dataout when cpu_addr(15 downto 14) = "00"
cpu0_datain <= "0000000" & cpu_int_vector when cpu_iako = '1'
			else sd0ram_dataout when cpu_addr(15 downto 14) = "00"
			else dp1ram_dataout when cpu_addr(15 downto 14) = "01"
			else rom0_dataout when cpu_addr(15 downto 13) = "100"
			else rom1_dataout when cpu_addr(15 downto 13) = "101"
			else rom02_dataout when cpu_addr(15 downto 13) = "110"
			else rom303_dataout when cpu_addr(15 downto 12) = "1110"
			else rom303_dataout when cpu_addr(15 downto 11) = "11110"
			else rom303_dataout when cpu_addr(15 downto 10) = "111110"
			else rom303_dataout when cpu_addr(15 downto 9) = "1111110"
			else rom303_dataout when cpu_addr(15 downto 8) = "11111110"
			else rom303_dataout when cpu_addr(15 downto 7) = "111111110"
			else sysregs_dataout when cpu_addr(15 downto 7) = "111111111"
			else (others=>'0');

---cpu1_datain <= sd2ram_dataout when cpu_addr(15 downto 14) = "00"
cpu1_datain <= "0000000" & cpu_int_vector when cpu_iako = '1'
			else sd2ram_dataout when cpu_addr(15 downto 14) = "00"
			else dp3ram_dataout when cpu_addr(15 downto 14) = "01"
			else rom0_dataout when cpu_addr(15 downto 13) = "100"
			else rom1_dataout when cpu_addr(15 downto 13) = "101"
			else rom12_dataout when cpu_addr(15 downto 13) = "110"
			else rom313_dataout when cpu_addr(15 downto 12) = "1110"
			else rom313_dataout when cpu_addr(15 downto 11) = "11110"
			else rom313_dataout when cpu_addr(15 downto 10) = "111110"
			else rom313_dataout when cpu_addr(15 downto 9) = "1111110"
			else rom313_dataout when cpu_addr(15 downto 8) = "11111110"
			else rom313_dataout when cpu_addr(15 downto 7) = "111111110"
			else sysregs_dataout when cpu_addr(15 downto 7) = "111111111"
			else (others=>'0');

---cpu2_datain <= sd4ram_dataout when cpu_addr(15 downto 14) = "00"
cpu2_datain <= "0000000" & cpu_int_vector when cpu_iako = '1'
			else sd4ram_dataout when cpu_addr(15 downto 14) = "00"
			else dp5ram_dataout when cpu_addr(15 downto 14) = "01"
			else rom0_dataout when cpu_addr(15 downto 13) = "100"
			else rom1_dataout when cpu_addr(15 downto 13) = "101"
			else rom22_dataout when cpu_addr(15 downto 13) = "110"
			else rom323_dataout when cpu_addr(15 downto 12) = "1110"
			else rom323_dataout when cpu_addr(15 downto 11) = "11110"
			else rom323_dataout when cpu_addr(15 downto 10) = "111110"
			else rom323_dataout when cpu_addr(15 downto 9) = "1111110"
			else rom323_dataout when cpu_addr(15 downto 8) = "11111110"
			else rom323_dataout when cpu_addr(15 downto 7) = "111111110"
			else sysregs_dataout when cpu_addr(15 downto 7) = "111111111"
			else (others=>'0');

mk_cpu_addr:
process (vgaclk,reset,cpu0_ad_out,cpu_sync)
begin
	if reset = '1' then
		cpu_addr <= x"0000" after 1 ns;
		cpu_dsync <= '0' after 1 ns;
	elsif vgaclk'event and vgaclk = '1' then
		cpu_dsync <= cpu_sync after 1 ns;
		if (cpu_sync and not cpu_dsync) = '1' then
			cpu_addr <= cpu0_ad_out after 1 ns;
		end if;
	end if;
end process;

mk_sbasic:
process (vgaclk,reset)
begin
	if vgaclk'event and vgaclk = '1' then
		if reset = '1' then
			sbasic <= not basic after 1 ns;
		end if;
	end if;
end process;

process (vgaclk,reset)
begin
	if reset = '1' then
---		swcnt <= "0000010" after 1 ns;
		swcnt <= "0000111" after 1 ns;
---		swcnt <= "1111111" after 1 ns;
	elsif vgaclk'event and vgaclk = '1' then
		if swcnt = "0000000" then
---			swcnt <= "0000010" after 1 ns; --- 50 MHz / 3 = 16,(6) MHz
			swcnt <= "0000111" after 1 ns; --- 25 MHz / 8 = 3,125 MHz
---			swcnt <= "1111111" after 1 ns; --- 50 MHz / 64 =  MHz
		else
			swcnt <= swcnt - "1" after 1 ns;
		end if;
	end if;
end process;
	
vgaclken <= '1' when swcnt = "0000000" else '0';

process (cpuclkn,cpuclken,skbdstop,reset)
begin
	if reset = '1' then
		sclkcnt <= "01111" after 1 ns;
		skbd1stop <= '0' after 1 ns;
		skbd2stop <= '0' after 1 ns;
		sintstop <= '0' after 1 ns;
	elsif cpuclkn'event and cpuclkn = '0' then
		if sclkcnt = "00000" then
			sclkcnt <= "01111" after 1 ns; --- 2,5 MHz / 16 = 0,156250 MHz
		else
			sclkcnt <= sclkcnt - "1" after 1 ns;
		end if;
		if cpuclken = '1' then
			skbd1stop <= skbdstop after 1 ns;
			skbd2stop <= skbd1stop after 1 ns;
			sintstop <= skbd1stop and not skbd2stop after 1 ns;
		end if;
	end if;
end process;
	
cpuclken <= '1' when sclkcnt = "00000" else '0';

--sdram0: sdram port map(
--	CLK_I => sysclk,
--	PRES_I => cpu_init,
--	-- Memory port
--	ADDR_I => sdram_addr,
--	DATA_I => cpu0_dataout,
--	DATA_O => sdram_dataout,
--	WR_I => sdram_wr,
--	BE_I => cpu_be,
--	RD_I => sdram_rd,
--	RFSH_I => '0',
--	IDLE_O => open,
--	-- SDRAM Pin
--	CLK_O => DRAM_CLK,
--	RAS_O => DRAM_NRAS,
--	CAS_O => DRAM_NCAS,
--	WE_O => DRAM_NWE,
--	DQM_O => DRAM_DQM,
--	BA_O => DRAM_BA,
--	MA_O => DRAM_A,
--	DQ_IO => DRAM_DQ
--	);

dpram0: dp0ram16x8k port map(
	byteena_a => cpu_be,
	clock => sysclk,
	data_a => dp0_datain,
	data_b => (others=>'0'),
	address_a => cpu_addr(13 downto 1),
	address_b => "0000000000000",
	wren_a => sdram_wr,
	wren_b => '0',
	q_a => sd0ram_dataout,
	q_b => open
	);
   
dpram1: dp1ram16x8k port map(
	byteena_a => cpu_be,
	clock => sysclk,
	data_a => dp0_datain,
	data_b => (others=>'0'),
	address_a => cpu_addr(13 downto 1),
	address_b => vga_addr(12 downto 0),
	wren_a => dpram_wr,
	wren_b => '0',
	q_a => dp1ram_dataout,
	q_b => dp1vga_dataout
	);
   
dpram2: dp2ram16x8k port map(
	byteena_a => cpu_be,
	clock => sysclk,
	data_a => dp1_datain,
	data_b => (others=>'0'),
	address_a => cpu_addr(13 downto 1),
	address_b => "0000000000000",
	wren_a => sdram_wr,
	wren_b => '0',
	q_a => sd2ram_dataout,
	q_b => open
	);
   
dpram3: dp3ram16x8k port map(
	byteena_a => cpu_be,
	clock => sysclk,
	data_a => dp1_datain,
	data_b => (others=>'0'),
	address_a => cpu_addr(13 downto 1),
	address_b => vga_addr(12 downto 0),
	wren_a => dpram_wr,
	wren_b => '0',
	q_a => dp3ram_dataout,
	q_b => dp3vga_dataout
	);
   
dpram4: dp4ram16x8k port map(
	byteena_a => cpu_be,
	clock => sysclk,
	data_a => dp2_datain,
	data_b => (others=>'0'),
	address_a => cpu_addr(13 downto 1),
	address_b => "0000000000000",
	wren_a => sdram_wr,
	wren_b => '0',
	q_a => sd4ram_dataout,
	q_b => open
	);
  
dpram5: dp5ram16x8k port map(
	byteena_a => cpu_be,
	clock => sysclk,
	data_a => dp2_datain,
	data_b => (others=>'0'),
	address_a => cpu_addr(13 downto 1),
	address_b => vga_addr(12 downto 0),
	wren_a => dpram_wr,
	wren_b => '0',
	q_a => dp5ram_dataout,
	q_b => dp5vga_dataout
	);
   
rom0: rom016x4k port map(
	address => cpu_addr(12 downto 1),
	clock => vgaclk,
	q => rom0_dataout
	);
   
rom1: rom116x4k port map(
	address => cpu_addr(12 downto 1),
	clock => vgaclk,
	q => rom1_dataout
	);
   
rom3: rom316x4k port map(
	address => cpu_addr(12 downto 1),
	clock => vgaclk,
	q => rom3_dataout
	);
   
rom02: rom0216x4k port map(
	address => cpu_addr(12 downto 1),
	clock => vgaclk,
	q => rom02_dataout
	);
   
rom12: rom1216x4k port map(
	address => cpu_addr(12 downto 1),
	clock => vgaclk,
	q => rom12_dataout
	);
   
rom22: rom2216x4k port map(
	address => cpu_addr(12 downto 1),
	clock => vgaclk,
	q => rom22_dataout
	);
   
rom03: rom0316x4k port map(
	address => cpu_addr(12 downto 1),
	clock => vgaclk,
	q => rom03_dataout
	);

rom13: rom1316x4k port map(
	address => cpu_addr(12 downto 1),
	clock => vgaclk,
	q => rom13_dataout
	);

rom23: rom2316x4k port map(
	address => cpu_addr(12 downto 1),
	clock => vgaclk,
	q => rom23_dataout
	);

--rom111: rom1basic port map(
--	address => cpu_addr(12 downto 1),
--	clock => cpuclk,
--	q => basic1_dataout
--	);
--   
--rom222: rom2basic port map(
--	address => cpu_addr(12 downto 1),
--	clock => cpuclk,
--	q => basic2_dataout
--	);
--   
--rom333: rom3basic port map(
--	address => cpu_addr(12 downto 1),
--	clock => cpuclk,
--	q => basic3_dataout
--	);

ps20: bkkbd port map(
	base_addr => o"177660",

	addr_match => ps2_addr_match,
	addr => bus_addr(17 downto 0),
	dout => ps2_dataout,
	opalnum => spalnum,
	din => cpu0_dataout,
 	rd => cpu_rd,
	wr => cpu_wr,
 	dw8 => cpu_dw8,

	br => cpu_irqu,
	bg => cpu_iako,
	int_vector => cpu_int_vector,

	keyupdn => keyupdn,
	kbdreset => kbdreset,
	kbdcolor => kbdcolor,
	kbdsimd => kbdsimd,
	
	kbdstop => skbdstop,

	kbci => ps2k_c,
	kbdi => ps2k_d,

	TEST => sTEST,
	
	pres => cpu_init,
---	clken => vgaclken,
	clken => '1',
	clk => vgaclk
	);

---sborder <= '0' & port_i716_reg(4) & '0'; --- GBR (G=0, B=RX, R=0)
sborder <= '0' & port_i716_reg(4) & port_i716_reg(5); --- GBR (G=0, B=RX, R=tape_in)

vga0: bk1610vga port map(
	pres => cpu_init, --- active high
	aclk => sysclk, --- 100 MHz
	bk1vdata => dp1vga_dataout, --- BK 2 port VRAM data
	bk3vdata => dp3vga_dataout, --- BK 2 port VRAM data
	bk5vdata => dp5vga_dataout, --- BK 2 port VRAM data
	border => sborder,
	icolor => scolor,
	isimd => ssimd,
	ipalnum => spalnum,
---	vgaport => port_664_reg, --- vga port
	bkvaddr => vga_addr(15 downto 0), --- address of BK VRAM
	blank => svgablank, --- active high
	vgar => svgar, --- R
	vgag => svgag, --- G
	vgab => svgab, --- B 
	vsync => svgav, --- active low
	hsync => svgah, --- active low

	base_addr => o"177664",
	addr_match => vgacr_addr_match,
	addr => bus_addr(17 downto 0),
	dout => port_664_reg,
	din => cpu0_dataout,
	clken => vgaclken,
 	rd => cpu_rd,
	wr => cpu_wr,

	oInt => open --- active high
	);

-- HDMI
hdmi0: entity work.hdmi
port map(
	CLK_DVI_I	=> hdmiclk,
	CLK_PIXEL_I	=> vgaclk,
	R_I			=> svgar,
	G_I			=> svgag,
	B_I			=> svgab,
	BLANK_I		=> svgablank,
	HSYNC_I		=> svgah,
	VSYNC_I		=> svgav,
	HDMI_D0_O	=> sHDMI_D0,
	HDMI_D0N_O	=> sHDMI_D0N,
	HDMI_D1_O	=> sHDMI_D1,
	HDMI_D1N_O	=> sHDMI_D1N,
	HDMI_D2_O	=> sHDMI_D2,
	HDMI_D2N_O	=> sHDMI_D2N,
	HDMI_CLK_O	=> sHDMI_CLK,
	HDMI_CLKN_O	=> sHDMI_CLKN
	);

HDMI_D0  <= sHDMI_D0;
HDMI_D0N <= sHDMI_D0N;
HDMI_D1  <= sHDMI_D1;
HDMI_D1N <= sHDMI_D1N;
HDMI_D2  <= sHDMI_D2;
HDMI_D2N <= sHDMI_D2N;
HDMI_CLK  <= sHDMI_CLK;
HDMI_CLKN <= sHDMI_CLKN;

bus_addr <= "00" & cpu_addr;
---test_bus_addr <= bus_addr;
---TEST <= illhalt & ps2_dataout(6 downto 0);
---TEST <= kbdreset & ps2_dataout(6 downto 0);
---TEST <= '1' & ps2_dataout(6 downto 0);
---TEST <= port_664_reg(7 downto 0);
---TEST <= adc_d(11 downto 4);
---TEST <= "00000000";
---TEST <= "000000" & ssimd & scolor;
---TEST <= cpu_addr(14 downto 7);
TEST <= sTEST(7 downto 0);

sysregs_dataout <= ps2_dataout when cpu_addr(6 downto 2) = "01100"	--- o177660-o177662 kbd
				else "000000" & port_664_reg(9) & '0' & port_664_reg(7 downto 0) when cpu_addr(6 downto 1) = "011010"	--- o177664 vga port
				else port_700_reg when cpu_addr(6 downto 1) = "100000"	--- o177700  port
				else port_i702_reg when cpu_addr(6 downto 1) = "100001"	--- o177702  port
				else port_704_reg when cpu_addr(6 downto 1) = "100010"	--- o177704  port
				else port_i714_reg when cpu_addr(6 downto 1) = "100110"	--- o177714 io port
				else port_i716_reg when cpu_addr(6 downto 1) = "100111"	--- o177716 sys port
				else x"0000";
-----------------------------------------------------
--- SYSREGS

--- o177700
process (cpu_init,vgaclk,vgaclken,bus_addr,cpu_wr,cpu0_dataout)
begin
	if (cpu_init = '1') then
		port_700_reg <= x"FFE0"; --- o177740
	elsif (vgaclk'event and vgaclk = '1') then
		if vgaclken = '1' then
			if (cpu_wr = '1' and bus_addr(15 downto 0) = o"177700") then
				port_700_reg(4 downto 0) <= cpu0_dataout(4 downto 0);
			end if;
		end if;
	end if;
end process;
--- D[4:3] = 00	 PA1,PA0
--- D2 = 0   WT
--- D1 = 0   ST
--- D0 = 0   HLT
	 
--- o177702
process (cpu_init,vgaclk,vgaclken,bus_addr,cpu_wr,cpu0_dataout)
begin
	if (cpu_init = '1') then
		port_i702_reg <= x"FFFF";
		port_o702_reg <= x"0008";
	elsif (vgaclk'event and vgaclk = '1') then
		if vgaclken = '1' then
			if (cpu_wr = '1' and bus_addr(15 downto 0) = o"177702") then
				port_o702_reg <= cpu0_dataout;
			end if;
		end if;
	end if;
end process;
--- Ynicky --- oD3 - '1' - ROM03_13_23_33 / '0' - ROM3

--- o177704
process (cpu_init,vgaclk,vgaclken,bus_addr,cpu_wr,cpu0_dataout)
begin
	if (cpu_init = '1') then
		port_704_reg <= x"FF20";
	elsif (vgaclk'event and vgaclk = '1') then
		if vgaclken = '1' then
			if (cpu_wr = '1' and bus_addr(15 downto 0) = o"177704") then
				port_704_reg(7 downto 0) <= cpu0_dataout(7 downto 6) & '1' & cpu0_dataout(4 downto 0);
			end if;
		end if;
	end if;
end process;
--- D7 = 0   ER7	 reserved
--- D6 = 0   OAT	 odd address trap
--- D5 = 1
--- D4 = 0   QTO	 qbus timeout
--- D3 = 0   ER3	 reserved
--- D2 = 0   ER2	 reserved
--- D1 = 0   UOP	 unknown opcode
--- D0 = 0   DBL	 double error

--- o177714
process (cpu_init,vgaclk,vgaclken,bus_addr,cpu_wr,cpu0_dataout)
begin
	if (cpu_init = '1') then
		port_i714_reg <= (others => '0');
		port_o714_reg <= (others => '0');
	elsif (vgaclk'event and vgaclk = '1') then
		if vgaclken = '1' then
			if (cpu_wr = '1' and bus_addr(15 downto 0) = o"177714") then
				port_o714_reg <= cpu0_dataout;
			end if;
		end if;
	end if;
end process;

--- o177716
process (cpu_init,vgaclk,vgaclken,bus_addr,cpu_wr,cpu0_dataout)
begin
	if (cpu_init = '1') then
		port_i716_reg <= x"80E0"; --- o100340
		port_o716_reg <= x"8090"; --- o100220
	elsif (vgaclk'event and vgaclk = '1') then
		port_i716_reg(4) <= rx after 1 ns;
		port_i716_reg(5) <= '1' after 1 ns; --- MAG in
---		port_i716_reg(5) <= tape_in after 1 ns; --- MAG in
		port_i716_reg(6) <= keyupdn after 1 ns; --- '0' - Key down / '1' - Key up
		port_i716_reg(7) <= '1' after 1 ns;
		port_i716_reg(15 downto 8) <= x"80" after 1 ns; --- o100000
---test		port_i716_reg(15 downto 8) <= x"00" after 1 ns; --- test
		if vgaclken = '1' then
			if (cpu_wr = '1' and bus_addr(15 downto 0) = o"177716") then
---				port_o716_reg(7 downto 0) <= cpu0_dataout(7 downto 0) after 1 ns;
				port_o716_reg(7 downto 4) <= cpu0_dataout(7 downto 4) after 1 ns;
				port_i716_reg(2) <= '1' after 1 ns;
			elsif (cpu_rd = '1' and bus_addr(15 downto 0) = o"177716") then
				port_i716_reg(2) <= '0' after 1 ns;
			end if;
		end if;
	end if;
end process;
--- oD4 - TLG TX ('1')
--- oD5 - MAG out / TLG gotov (DTR)
--- oD6 - MAG out / Key down
--- oD7 - '0' - MAG start / '1'	- MAG stop ('1')
--- oD11 - '0' (bk1610)	/ '1' (BK0011)
--- iD4 - TLG RX ('1')
--- iD5 - MAG in
--- iD6 - '0' - Key down / '1' - Key up
--- iD7 - TLG gotov (CTS)???
--- iD7 - '1'

tx <= port_o716_reg(4);
DAC_OUT_L <= port_o716_reg(6);
DAC_OUT_R <= port_o716_reg(6);
-----------------------------------------------------
process (vgaclk,cpu_init,kbdcolor,vgaclken)
begin
	if (cpu_init = '1') then
		scolor <= '0';
		svcolor <= '0';
	elsif (vgaclk'event and vgaclk = '1') then
		if vgaclken = '1' then
			svcolor <= kbdcolor;
			if (kbdcolor and not svcolor) = '1' then
				scolor <= not scolor;
			end if;
		end if;
	end if;
end process;

process (vgaclk,cpu_init,kbdsimd,vgaclken)
begin
	if (cpu_init = '1') then
		ssimd <= '0';
		svsimd <= '0';
	elsif (vgaclk'event and vgaclk = '1') then
		if vgaclken = '1' then
			svsimd <= kbdsimd;
			if (kbdsimd and not svsimd) = '1' then
				ssimd <= not ssimd;
			end if;
		end if;
	end if;
end process;


	reset <= kbdreset or (not nreset) or (not locked);
	
---	ram_addr_match <= '1' when cpu_addr(15 downto 14) = "00"
---							else '0';
---	vga_addr_match <= '1' when cpu_addr(15 downto 14) = "01"
---							else '0';
	vgacr_addr_match <= '1' when cpu_addr(15 downto 0) = x"FFB4" --- o177664
							else '0';
	
end implementation;

